import { HttpErrorResponse } from '@angular/common/http';
import { Injectable } from '@angular/core';
import { Action, Store } from '@ngrx/store';
import { Observable, throwError } from 'rxjs';
import { catchError, filter, map } from 'rxjs/operators';

import { QuestionStateViewModel } from 'src/app/shared-app/models';
import { EffectStartedWorking, GetQuestionState } from './question-state/question.actions';
import { questionStateQuery } from './question-state/question.selectors';
import { QuestionState } from './question-state/question.state';

@Injectable({
  providedIn: 'root'
})
export class QuestionFacadeService {
  effectWorking$: Observable<boolean | HttpErrorResponse>;
  question$: Observable<QuestionStateViewModel>;

  constructor(private store: Store<QuestionState>) {
    this.question$ = this.store
      .select(questionStateQuery.getQuestion)
      .pipe(filter((x: QuestionStateViewModel | null): x is QuestionStateViewModel => x !== null));
    this.effectWorking$ = this.store.select(questionStateQuery.effectWorking).pipe(
      filter((effect) => effect !== null),
      map((effect) => {
        if (effect instanceof HttpErrorResponse) {
          throw effect;
        } else {
          return effect;
        }
      }),
      catchError((err) => {
        return throwError(err);
      })
    );
  }

  public fetchQuestion(questionUid: string): void {
    this.dispatchEffect(new GetQuestionState(questionUid));
  }

  private dispatch(action: Action): void {
    this.store.dispatch(action);
  }

  private dispatchEffect(action: Action): void {
    this.dispatch(new EffectStartedWorking());
    this.dispatch(action);
  }
}
