import { Injectable } from '@angular/core';
import { User, UserManager, WebStorageStateStore } from 'oidc-client';
import { Observable, of } from 'rxjs';

import { environment } from '@env/environment';
import { ApplicationUser, ApplicationUserType, SelfUserResponse } from 'src/app/shared-app/models';
import { BaseApiService } from 'src/app/shared-app/services/base-api.service';

@Injectable({
  providedIn: 'root'
})
export class AuthService {
  private userManager: UserManager = new UserManager({
    authority: environment.identityRoute,
    client_id: environment.identityClientId,
    redirect_uri: `${window.location.origin}/auth-callback`,
    response_type: 'id_token token',
    scope: 'openid app.api.finki-chattery profile',
    post_logout_redirect_uri: window.location.origin,
    filterProtocolClaims: true,
    loadUserInfo: true,
    userStore: new WebStorageStateStore({ store: window.localStorage })
  });

  public user: ApplicationUser | null = null;
  public oidcUser: User | null = null;

  constructor(private baseApi: BaseApiService) {
    this.userManager.getUser().then((user) => {
      this.oidcUser = user;
      this.user = new ApplicationUser(
        user?.profile.id,
        user?.profile.userType,
        user?.profile.emailAddress,
        user?.profile.username,
        user?.profile.isVerified
      );
    });
  }

  public login(): void {
    this.userManager.signinRedirect();
  }

  public logout(): void {
    this.userManager.signoutRedirect();
  }

  public isLoggedIn(): boolean {
    if (this.oidcUser) {
      return !this.oidcUser.expired;
    }
    return false;
  }

  public isStudent(): boolean {
    return this.user?.userType === ApplicationUserType.Student;
  }

  public currentUser(): ApplicationUser | null {
    return this.user;
  }

  public currentUserToken(): string {
    if (this.oidcUser) {
      return this.oidcUser.access_token;
    }

    return '';
  }

  public selfUserDto(): Observable<SelfUserResponse | null> {
    if (this.isLoggedIn()) {
      return this.baseApi.getSelfUser();
    }
    return of(null);
  }

  public async completeAuthentication(): Promise<void> {
    return await this.userManager.signinRedirectCallback().then((user: User) => {
      this.oidcUser = user;
      this.user = new ApplicationUser(
        user.profile.id,
        user.profile.userType,
        user.profile.emailAddress,
        user.profile.username,
        user.profile.isVerified
      );
    });
  }
}
