package mk.ukim.finki.it.reservengo.service.impl;

import mk.ukim.finki.it.reservengo.dto.userDTO.*;
import mk.ukim.finki.it.reservengo.model.domain.User;
import mk.ukim.finki.it.reservengo.model.exceptions.EmailNotFoundException;
import mk.ukim.finki.it.reservengo.model.exceptions.UserEmailAlreadyExistsException;
import mk.ukim.finki.it.reservengo.model.exceptions.UserIdNotFoundException;
import mk.ukim.finki.it.reservengo.repository.UserRepository;
import mk.ukim.finki.it.reservengo.service.intf.JWTService;
import mk.ukim.finki.it.reservengo.service.intf.UserService;
import org.springframework.security.crypto.password.PasswordEncoder;
import org.springframework.stereotype.Service;

@Service
public class UserServiceImpl implements UserService {

    private final UserRepository userRepository;
    private final JWTService jwtService;
    private final PasswordEncoder passwordEncoder;

    public UserServiceImpl(UserRepository userRepository, PasswordEncoder passwordEncoder, JWTService jwtService) {
        this.userRepository = userRepository;
        this.jwtService = jwtService;
        this.passwordEncoder = passwordEncoder;
    }

    @Override
    public User getUserById(Long userId) {
        return userRepository.findById(userId).orElseThrow(() -> new UserIdNotFoundException(userId));
    }

    @Override
    public User getUserByEmail(String email) {
        return userRepository.findByEmail(email).orElseThrow(() -> new EmailNotFoundException(email));
    }

    @Override
    public boolean emailExists(String email) {
        return userRepository.findByEmail(email).isPresent();
    }

    @Override
    public DisplayUserEmailDTO changeEmail(Long userId, EditUserEmailDTO editUserEmailDTO) {
        User user = userRepository.findById(userId).orElseThrow(() -> new UserIdNotFoundException(userId));

        if (editUserEmailDTO.newEmail().equals(user.getEmail())) {
            throw new IllegalArgumentException("New email must be different from the current email.");
        }
        if (emailExists(editUserEmailDTO.newEmail())) {
            throw new UserEmailAlreadyExistsException(editUserEmailDTO.newEmail());
        }
        user.setEmail(editUserEmailDTO.newEmail());

        userRepository.save(user);
        String jwt = jwtService.generateToken(user);

        return DisplayUserEmailDTO.fromUser(user, jwt);
    }

    @Override
    public String changePassword(Long userId, EditUserPasswordDTO editUserPasswordDTO) {
        User user = userRepository.findById(userId).orElseThrow(() -> new UserIdNotFoundException(userId));

        if (!passwordEncoder.matches(editUserPasswordDTO.currentPassword(), user.getPassword())) {
            throw new IllegalArgumentException("Current password is incorrect");
        }

        user.setPassword(passwordEncoder.encode(editUserPasswordDTO.newPassword()));
        userRepository.save(user);

        return "Password updated successfully!";
    }

    @Override
    public DisplayUserDTO editUser(Long userId, EditUserProfileDTO editUserProfileDTO) {
        User user = userRepository.findById(userId).orElseThrow(() -> new UserIdNotFoundException(userId));

        user.setFirstName(editUserProfileDTO.firstName());
        user.setLastName(editUserProfileDTO.lastName());
        user.setPhoneNumber(editUserProfileDTO.phoneNumber());
        userRepository.save(user);

        return DisplayUserDTO.fromUser(user);
    }
}
