package mk.ukim.finki.it.reservengo.service.impl;

import jakarta.mail.MessagingException;
import mk.ukim.finki.it.reservengo.dto.workerDTO.DisplayLocalReservationDTO;
import mk.ukim.finki.it.reservengo.model.domain.Local;
import mk.ukim.finki.it.reservengo.model.domain.LocalWorker;
import mk.ukim.finki.it.reservengo.model.domain.Reservation;
import mk.ukim.finki.it.reservengo.model.enumerations.Position;
import mk.ukim.finki.it.reservengo.model.enumerations.ReservationStatus;
import mk.ukim.finki.it.reservengo.model.enumerations.Role;
import mk.ukim.finki.it.reservengo.model.exceptions.*;
import mk.ukim.finki.it.reservengo.repository.LocalRepository;
import mk.ukim.finki.it.reservengo.repository.LocalWorkerRepository;
import mk.ukim.finki.it.reservengo.repository.ReservationRepository;
import mk.ukim.finki.it.reservengo.service.intf.EmailService;
import mk.ukim.finki.it.reservengo.service.intf.LocalWorkerService;
import mk.ukim.finki.it.reservengo.service.intf.SmsService;
import mk.ukim.finki.it.reservengo.service.intf.UserService;
import org.springframework.stereotype.Service;

import java.util.List;
import java.util.stream.Collectors;

@Service
public class LocalWorkerServiceImpl implements LocalWorkerService {
    private final LocalWorkerRepository localWorkerRepository;
    private final UserService userService;
    private final ReservationRepository reservationRepository;
    private final SmsService smsService;
    private final EmailService emailService;
    private final LocalRepository localRepository;


    public LocalWorkerServiceImpl(LocalWorkerRepository localWorkerRepository, UserService userService, ReservationRepository reservationRepository, SmsService smsService, EmailService emailService, LocalRepository localRepository) {
        this.localWorkerRepository = localWorkerRepository;
        this.userService = userService;
        this.reservationRepository = reservationRepository;
        this.smsService = smsService;
        this.emailService = emailService;
        this.localRepository = localRepository;
    }

    @Override
    public void save(LocalWorker localWorker) {
        if (userService.emailExists(localWorker.getEmail())) {
            throw new UserEmailAlreadyExistsException(localWorker.getEmail());
        }
        localWorkerRepository.save(localWorker);
    }

    @Override
    public LocalWorker findWorkerById(Long id) {
        return localWorkerRepository.findById(id).orElseThrow(() -> new WorkerIdNotFoundException(id));
    }

    @Override
    public void changePosition(LocalWorker worker, Position position) {
        worker.setPosition(position);
        localWorkerRepository.save(worker);
    }

    @Override
    public List<LocalWorker> findAllByLocalId(Long id) {
        return localWorkerRepository.findByLocal_Id(id);
    }

    @Override
    public List<LocalWorker> findAllUnassigned() {
        return localWorkerRepository.findAll().stream()
                .filter(worker -> worker.getLocal() == null)
                .filter(worker -> worker.getPosition() == null)
                .collect(Collectors.toList());
    }

    @Override
    public void updateLocalAssignment(LocalWorker worker, Local local) {
        worker.setLocal(local);
        if (local == null) {
            worker.setPosition(null);
        }
        localWorkerRepository.save(worker);
    }

    @Override
    public List<DisplayLocalReservationDTO> viewWorkerLocalReservations(Long workerUserId, ReservationStatus status) {
        LocalWorker localWorker = validateWorker(workerUserId);
        Local local = localWorker.getLocal();

        List<Reservation> reservations =
                (status != null)
                        ? reservationRepository.findByLocalIdAndStatus(local.getId(), status)
                        : reservationRepository.findByLocalId(local.getId());

        return DisplayLocalReservationDTO.from(reservations);
    }

    @Override
    public void acceptReservation(Long workerUserId, Long reservationId) throws MessagingException {
        LocalWorker worker = validateWorker(workerUserId);
        Reservation reservation = getAuthorizedReservation(worker, reservationId);

        reservation.setStatus(ReservationStatus.ACCEPTED);
        reservationRepository.save(reservation);

//        smsService.sendSms(
//                reservation.getCustomer().getPhoneNumber(),
//                "Your reservation has been ACCEPTED."
//        );

        emailService.sendReservationFeedbackEmail(reservation.getCustomer().getEmail(), ReservationStatus.ACCEPTED);
    }

    @Override
    public void denyReservation(Long workerUserId, Long reservationId) throws MessagingException {
        LocalWorker worker = validateWorker(workerUserId);
        Reservation reservation = getAuthorizedReservation(worker, reservationId);

        reservation.setStatus(ReservationStatus.CANCELED);
        reservationRepository.save(reservation);

        smsService.sendSms(
                reservation.getCustomer().getPhoneNumber(),
                "Your reservation has been DECLINED."
        );

        emailService.sendReservationFeedbackEmail(reservation.getCustomer().getEmail(), ReservationStatus.CANCELED);
    }

    @Override
    public void finishReservation(Long id, Long reservationId) {
        LocalWorker worker = validateWorker(id);
        Reservation reservation = getAuthorizedReservation(worker, reservationId);

        reservation.setStatus(ReservationStatus.FINISHED);
        reservationRepository.save(reservation);
    }

    @Override
    public void deleteReservation(Long id, Long reservationId) {
        LocalWorker worker = localWorkerRepository.findById(id).orElseThrow(() -> new WorkerIdNotFoundException(id));
        Reservation reservation = getAuthorizedReservation(worker, reservationId);
        Local local = worker.getLocal();
        reservation.setStatus(ReservationStatus.CANCELED);
        reservation.setLocal(null);
        local.getReservations().remove(reservation);
        reservationRepository.save(reservation);
        localRepository.save(local);
    }

    private Reservation getAuthorizedReservation(LocalWorker worker, Long reservationId) {
        Reservation reservation = reservationRepository.findById(reservationId)
                .orElseThrow(() -> new ReservationIdNotFoundException(reservationId));

        if (!reservation.getLocal().getId().equals(worker.getLocal().getId())) {
            throw new UnauthorizedException(worker.getId());
        }

        return reservation;
    }


    private LocalWorker validateWorker(Long workerUserId) {
        LocalWorker worker = localWorkerRepository.findById(workerUserId)
                .orElseThrow(() -> new WorkerIdNotFoundException(workerUserId));

        if (worker.getUserRole() != Role.ROLE_LOCAL_WORKER) {
            throw new UnauthorizedException(workerUserId);
        }

        if (worker.getLocal() == null) {
            throw new WorkerNotAssignedException(workerUserId);
        }

        return worker;
    }
}
