package mk.ukim.finki.it.reservengo.service.impl;

import mk.ukim.finki.it.reservengo.model.exceptions.InvalidFileException;
import mk.ukim.finki.it.reservengo.model.exceptions.PhotoDeletionException;
import mk.ukim.finki.it.reservengo.model.exceptions.PhotoUploadException;
import mk.ukim.finki.it.reservengo.service.intf.FileStorageService;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.web.multipart.MultipartFile;

import java.io.IOException;
import java.nio.file.*;
import java.util.Objects;
import java.util.Set;
import java.util.UUID;

@Service
@Transactional
public class FileStorageServiceImpl implements FileStorageService {

    private final Path rootUploadsDir = Paths.get("uploads");
    private final Path logosDir = rootUploadsDir.resolve("logos");
    private final Path localsDir = rootUploadsDir.resolve("locals");
    private final Path profilesDir = rootUploadsDir.resolve("profiles");
    private static final Set<String> ALLOWED_EXTENSIONS = Set.of("jpg", "jpeg", "png");
    private static final Set<String> ALLOWED_CONTENT_TYPES = Set.of("image/jpeg", "image/png");
    private static final long MAX_FILE_SIZE = 5 * 1024 * 1024; // 5MB

    public FileStorageServiceImpl() throws IOException {
        try {
            createIfNotExists(rootUploadsDir);
            createIfNotExists(logosDir);
            createIfNotExists(localsDir);
            createIfNotExists(profilesDir);
        } catch (IOException e) {
            throw new FileSystemException("Could not initialize storage directories");
        }
    }

    private void createIfNotExists(Path dir) throws IOException {
        if (!Files.exists(dir)) {
            Files.createDirectories(dir);
        }
    }

    @Override
    public String saveLogoFile(MultipartFile file) {
        return savePhotoFile(file, logosDir, "/uploads/logos/");
    }

    @Override
    public String saveProfilePhotoFile(MultipartFile file) {
        return savePhotoFile(file, profilesDir, "/uploads/profiles/");
    }

    @Override
    public String savePhotoFile(MultipartFile file) {
        return savePhotoFile(file, localsDir, "/uploads/locals/");
    }

    @Override
    public void deletePhotoFile(String filePath) {
        if (filePath == null || filePath.trim().isEmpty()) {
            return;
        }
        try {
            String normalizedPath = filePath.replaceFirst("^/uploads/?", "");
            Path fullPath = rootUploadsDir.resolve(normalizedPath).normalize();
            Files.deleteIfExists(fullPath);
        } catch (IOException e) {
            throw new PhotoDeletionException("Failed to delete photo.");
        }
    }

    private void validateFile(MultipartFile file) {
        if (file == null || file.isEmpty()) {
            throw new InvalidFileException("Photo file cannot be empty");
        }

        if (file.getSize() > MAX_FILE_SIZE) {
            throw new InvalidFileException("Photo file size exceeds maximum allowed size of 5MB");
        }

        String contentType = file.getContentType();
        if (contentType == null || !ALLOWED_CONTENT_TYPES.contains(contentType.toLowerCase())) {
            throw new InvalidFileException("Invalid file type. Only:" + ALLOWED_CONTENT_TYPES + " are allowed.");
        }

        String originalFilename = file.getOriginalFilename();
        if (originalFilename == null || originalFilename.trim().isEmpty()) {
            throw new InvalidFileException("Invalid filename");
        }

        String fileExtension = getFileExtension(originalFilename).toLowerCase();
        if (!ALLOWED_EXTENSIONS.contains(fileExtension)) {
            throw new InvalidFileException("File extension not allowed: " + fileExtension);
        }
    }

    private String generateUniqueFileName(String extension) {
        return UUID.randomUUID() + "." + extension;
    }

    private String getFileExtension(String filename) {
        int lastDotIndex = filename.lastIndexOf('.');
        return lastDotIndex > 0 ? filename.substring(lastDotIndex + 1) : "";
    }

    private String savePhotoFile(MultipartFile file, Path targetDir, String webPathPrefix) {
        validateFile(file);
        String originalFilename = file.getOriginalFilename();
        String fileExtension = getFileExtension(Objects.requireNonNull(originalFilename)).toLowerCase();

        try {
            String fileName = generateUniqueFileName(fileExtension);
            Path targetPath = targetDir.resolve(fileName);
            Files.copy(file.getInputStream(), targetPath, StandardCopyOption.REPLACE_EXISTING);
            return webPathPrefix + fileName;
        } catch (IOException e) {
            throw new PhotoUploadException("Failed to upload photo.");
        }
    }
}
