package mk.ukim.finki.it.reservengo.service.impl;

import mk.ukim.finki.it.reservengo.dto.jwtDTO.JWTAuthenticationRequestDTO;
import mk.ukim.finki.it.reservengo.dto.jwtDTO.JWTAuthenticationResponseDTO;
import mk.ukim.finki.it.reservengo.dto.jwtDTO.JWTLoginDTO;
import mk.ukim.finki.it.reservengo.model.domain.Customer;
import mk.ukim.finki.it.reservengo.model.domain.LocalManager;
import mk.ukim.finki.it.reservengo.model.domain.LocalWorker;
import mk.ukim.finki.it.reservengo.model.domain.User;
import mk.ukim.finki.it.reservengo.model.exceptions.EmailNotFoundException;
import mk.ukim.finki.it.reservengo.repository.UserRepository;
import mk.ukim.finki.it.reservengo.service.intf.*;
import org.springframework.security.authentication.AuthenticationManager;
import org.springframework.security.authentication.UsernamePasswordAuthenticationToken;
import org.springframework.security.crypto.password.PasswordEncoder;
import org.springframework.stereotype.Service;

@Service
public class AuthServiceImpl implements AuthService {
    private final CustomerService customerService;
    private final LocalWorkerService localWorkerService;
    private final LocalManagerService localManagerService;
    private final JWTService jwtService;
    private final PasswordEncoder passwordEncoder;
    private final AuthenticationManager authenticationManager;
    private final UserRepository userRepository;

    public AuthServiceImpl(CustomerService customerService, LocalWorkerService localWorkerService, LocalManagerService localManagerService, JWTService jwtService, PasswordEncoder passwordEncoder, AuthenticationManager authenticationManager, UserRepository userRepository) {
        this.customerService = customerService;
        this.localWorkerService = localWorkerService;
        this.localManagerService = localManagerService;
        this.jwtService = jwtService;
        this.passwordEncoder = passwordEncoder;
        this.authenticationManager = authenticationManager;
        this.userRepository = userRepository;
    }

    @Override
    public JWTAuthenticationResponseDTO registerCustomer(JWTAuthenticationRequestDTO jwtAuthenticationRequestDTO) {
        Customer customer = jwtAuthenticationRequestDTO.toCustomer(passwordEncoder.encode(jwtAuthenticationRequestDTO.password()));

        customerService.save(customer);
        String jwt = jwtService.generateToken(customer);

        return JWTAuthenticationResponseDTO.fromUser(customer, jwt);
    }

    @Override
    public JWTAuthenticationResponseDTO registerLocalWorker(JWTAuthenticationRequestDTO jwtAuthenticationRequestDTO) {
        LocalWorker localWorker = jwtAuthenticationRequestDTO.toLocalWorker(passwordEncoder.encode(jwtAuthenticationRequestDTO.password()));

        localWorkerService.save(localWorker);
        String jwt = jwtService.generateToken(localWorker);

        return JWTAuthenticationResponseDTO.fromUser(localWorker, jwt);
    }

    @Override
    public JWTAuthenticationResponseDTO registerLocalManager(JWTAuthenticationRequestDTO jwtAuthenticationRequestDTO) {
        LocalManager localManager = jwtAuthenticationRequestDTO.toLocalManager(passwordEncoder.encode(jwtAuthenticationRequestDTO.password()));

        localManagerService.save(localManager);
        String jwt = jwtService.generateToken(localManager);

        return JWTAuthenticationResponseDTO.fromUser(localManager, jwt);
    }

    @Override
    public JWTAuthenticationResponseDTO login(JWTLoginDTO jwtLoginDTO) {
        authenticationManager.authenticate(new UsernamePasswordAuthenticationToken(jwtLoginDTO.email(), jwtLoginDTO.password()));

        User user = userRepository.findByEmail(jwtLoginDTO.email()).orElseThrow(() -> new EmailNotFoundException(jwtLoginDTO.email()));
        String jwt = jwtService.generateToken(user);

        return JWTAuthenticationResponseDTO.fromUser(user, jwt);
    }
}
