package mk.ukim.finki.it.reservengo.service.impl;

import mk.ukim.finki.it.reservengo.dto.localDTO.DisplayLocalDTO;
import mk.ukim.finki.it.reservengo.model.domain.Customer;
import mk.ukim.finki.it.reservengo.model.domain.Local;
import mk.ukim.finki.it.reservengo.model.exceptions.CustomerIdNotFoundException;
import mk.ukim.finki.it.reservengo.model.exceptions.LocalAlreadyFavouredException;
import mk.ukim.finki.it.reservengo.model.exceptions.LocalAlreadyUnfavouredException;
import mk.ukim.finki.it.reservengo.model.exceptions.UserEmailAlreadyExistsException;
import mk.ukim.finki.it.reservengo.repository.CustomerRepository;
import mk.ukim.finki.it.reservengo.service.intf.CustomerService;
import mk.ukim.finki.it.reservengo.service.intf.LocalService;
import mk.ukim.finki.it.reservengo.service.intf.UserService;
import org.springframework.stereotype.Service;

import java.util.List;

@Service
public class CustomerServiceImpl implements CustomerService {
    private final CustomerRepository customerRepository;
    private final LocalService localService;
    private final UserService userService;

    public CustomerServiceImpl(CustomerRepository customerRepository, LocalService localService, UserService userService) {
        this.customerRepository = customerRepository;
        this.localService = localService;
        this.userService = userService;
    }

    @Override
    public List<DisplayLocalDTO> listFavouriteLocals(Long id) {
        Customer customer = customerRepository.findById(id).orElseThrow(() -> new CustomerIdNotFoundException(id));
        return DisplayLocalDTO.fromLocals(customer.getFavouriteLocals());
    }

    @Override
    public void addFavouriteLocal(Long userId, Long localId) {
        Customer customer = customerRepository.findById(userId).orElseThrow(() -> new CustomerIdNotFoundException(userId));
        Local local = localService.findLocalById(localId);

        if (customer.getFavouriteLocals().contains(local)) {
            throw new LocalAlreadyFavouredException(localId);
        }

        customer.getFavouriteLocals().add(local);
        customerRepository.save(customer);
    }

    @Override
    public void removeFavouriteLocal(Long userId, Long localId) {
        Customer customer = customerRepository.findById(userId).orElseThrow(() -> new CustomerIdNotFoundException(userId));
        Local local = localService.findLocalById(localId);

        if (!customer.getFavouriteLocals().contains(local)) {
            throw new LocalAlreadyUnfavouredException(localId);
        }

        customer.getFavouriteLocals().remove(local);
        customerRepository.save(customer);
    }

    @Override
    public void save(Customer customer) {
        if (userService.emailExists(customer.getEmail())) {
            throw new UserEmailAlreadyExistsException(customer.getEmail());
        }
        customerRepository.save(customer);
    }
}
