package mk.ukim.finki.it.reservengo.service.impl;

import com.google.api.client.googleapis.auth.oauth2.GoogleIdToken;
import com.google.api.client.googleapis.auth.oauth2.GoogleIdTokenVerifier;
import com.google.common.base.VerifyException;
import jakarta.mail.MessagingException;
import mk.ukim.finki.it.reservengo.dto.jwtDTO.JWTAuthenticationRequestDTO;
import mk.ukim.finki.it.reservengo.dto.jwtDTO.JWTAuthenticationResponseDTO;
import mk.ukim.finki.it.reservengo.dto.jwtDTO.JWTLoginDTO;
import mk.ukim.finki.it.reservengo.model.domain.Customer;
import mk.ukim.finki.it.reservengo.model.domain.LocalManager;
import mk.ukim.finki.it.reservengo.model.domain.LocalWorker;
import mk.ukim.finki.it.reservengo.model.domain.User;
import mk.ukim.finki.it.reservengo.model.enumerations.Provider;
import mk.ukim.finki.it.reservengo.model.enumerations.Role;
import mk.ukim.finki.it.reservengo.model.exceptions.EmailNotFoundException;
import mk.ukim.finki.it.reservengo.repository.UserRepository;
import mk.ukim.finki.it.reservengo.service.intf.*;
import org.springframework.security.authentication.AuthenticationManager;
import org.springframework.security.authentication.BadCredentialsException;
import org.springframework.security.authentication.UsernamePasswordAuthenticationToken;
import org.springframework.security.crypto.password.PasswordEncoder;
import org.springframework.stereotype.Service;

import java.io.IOException;
import java.security.GeneralSecurityException;
import java.time.LocalDateTime;
import java.util.Random;
import java.util.UUID;

@Service
public class AuthServiceImpl implements AuthService {
    private final CustomerService customerService;
    private final LocalWorkerService localWorkerService;
    private final LocalManagerService localManagerService;
    private final JWTService jwtService;
    private final PasswordEncoder passwordEncoder;
    private final AuthenticationManager authenticationManager;
    private final UserRepository userRepository;
    private final UserService userService;
    private final EmailService emailService;
    private final GoogleIdTokenVerifier googleIdTokenVerifier;

    public AuthServiceImpl(CustomerService customerService, LocalWorkerService localWorkerService, LocalManagerService localManagerService, JWTService jwtService, PasswordEncoder passwordEncoder, AuthenticationManager authenticationManager, UserRepository userRepository, UserService userService, EmailService emailService, GoogleIdTokenVerifier googleIdTokenVerifier) {
        this.customerService = customerService;
        this.localWorkerService = localWorkerService;
        this.localManagerService = localManagerService;
        this.jwtService = jwtService;
        this.passwordEncoder = passwordEncoder;
        this.authenticationManager = authenticationManager;
        this.userRepository = userRepository;
        this.userService = userService;
        this.emailService = emailService;
        this.googleIdTokenVerifier = googleIdTokenVerifier;
    }

    @Override
    public JWTAuthenticationResponseDTO registerCustomer(JWTAuthenticationRequestDTO jwtAuthenticationRequestDTO) throws MessagingException {
        Customer customer = jwtAuthenticationRequestDTO.toCustomer(passwordEncoder.encode(jwtAuthenticationRequestDTO.password()), Provider.LOCAL, null);

        String verificationCode = generateVerificationCode();
        customer.setVerificationCode(verificationCode);
        customer.setVerificationCodeExpiryDate(LocalDateTime.now().plusMinutes(30));
        customer.setEnabled(false);

        customerService.save(customer);

        emailService.sendVerificationEmail(customer.getEmail(), verificationCode);

        return JWTAuthenticationResponseDTO.fromUser(customer, null);
    }

    @Override
    public JWTAuthenticationResponseDTO registerLocalWorker(JWTAuthenticationRequestDTO jwtAuthenticationRequestDTO, String token) {
        validateTokenEmail(token, jwtAuthenticationRequestDTO.email());

        LocalWorker localWorker = jwtAuthenticationRequestDTO.toLocalWorker(passwordEncoder.encode(jwtAuthenticationRequestDTO.password()), Provider.LOCAL, null);

        localWorkerService.save(localWorker);
        String jwt = jwtService.generateToken(localWorker);

        return JWTAuthenticationResponseDTO.fromUser(localWorker, jwt);
    }

    @Override
    public JWTAuthenticationResponseDTO registerLocalManager(JWTAuthenticationRequestDTO jwtAuthenticationRequestDTO, String token) {
        validateTokenEmail(token, jwtAuthenticationRequestDTO.email());

        LocalManager localManager = jwtAuthenticationRequestDTO.toLocalManager(passwordEncoder.encode(jwtAuthenticationRequestDTO.password()), Provider.LOCAL, null);

        localManagerService.save(localManager);
        String jwt = jwtService.generateToken(localManager);

        return JWTAuthenticationResponseDTO.fromUser(localManager, jwt);
    }

    @Override
    public JWTAuthenticationResponseDTO login(JWTLoginDTO jwtLoginDTO) throws MessagingException {
        User user = userRepository.findByEmail(jwtLoginDTO.email()).orElseThrow(() -> new EmailNotFoundException(jwtLoginDTO.email()));

        if (user.getProvider() != Provider.LOCAL) {
            throw new MessagingException("This account must be logged in via " + user.getProvider() + ".");
        }

        if (!user.isEnabled() && user.getVerificationCodeExpiryDate() != null && user.getVerificationCode() != null) {
            throw new MessagingException("Account not verified. Please verify your email.");
        }

        authenticationManager.authenticate(new UsernamePasswordAuthenticationToken(jwtLoginDTO.email(), jwtLoginDTO.password()));
        String jwt = jwtService.generateToken(user);

        userService.updateUserActivity(user.getId());

        return JWTAuthenticationResponseDTO.fromUser(user, jwt);
    }

    @Override
    public void reactivateProfile(JWTLoginDTO jwtLoginDTO) {
        User user = userService.findUserByEmail(jwtLoginDTO.email());

        if (user == null) {
            throw new EmailNotFoundException(jwtLoginDTO.email());
        }

        if (!passwordEncoder.matches(jwtLoginDTO.password(), user.getPassword())) {
            throw new BadCredentialsException("Bad credentials.");
        }

        if (!user.isEnabled()) {
            userService.enableProfile(user.getId());
        }
    }

    @Override
    public void verifyCustomer(String email, String verificationCode) throws MessagingException {
        User user = userRepository.findByEmail(email).orElseThrow(() -> new EmailNotFoundException(email));

        if (user.getVerificationCodeExpiryDate().isBefore(LocalDateTime.now())) {
            throw new MessagingException("Verification code expired");
        }

        if (!user.getVerificationCode().equals(verificationCode.trim())) {
            throw new MessagingException("Invalid verification code");
        }

        user.setEnabled(true);
        user.setVerificationCode(null);
        user.setVerificationCodeExpiryDate(null);

        userRepository.save(user);
    }

    @Override
    public void resendVerificationCode(String email) throws MessagingException {
        User user = userRepository.findByEmail(email).orElseThrow(() -> new EmailNotFoundException(email));

        if (user.isEnabled()) {
            throw new MessagingException("Account already verified");
        }

        String newCode = generateVerificationCode();
        user.setVerificationCode(newCode);
        user.setVerificationCodeExpiryDate(LocalDateTime.now().plusMinutes(30));

        try {
            emailService.sendVerificationEmail(email, newCode);
        } catch (MessagingException e) {
            throw new MessagingException("Failed to send email", e);
        }

        userRepository.save(user);
    }

    @Override
    public JWTAuthenticationResponseDTO registerGoogleCustomer(String idTokenString) {
        GoogleIdToken.Payload payload = verifyGoogleIdTokenAndGetPayload(idTokenString);

        String email = payload.getEmail();
        String givenName = (String) payload.get("given_name");
        String familyName = (String) payload.get("family_name");
        String providerId = payload.getSubject();

        return userRepository.findByEmail(email)
                .map(user -> {
                    String jwt = jwtService.generateToken(user);
                    return JWTAuthenticationResponseDTO.fromUser(user, jwt);
                })
                .orElseGet(() -> {
                    Customer customer = new Customer(
                            givenName,
                            familyName,
                            email,
                            passwordEncoder.encode(UUID.randomUUID().toString()),
                            null,
                            Role.ROLE_CUSTOMER,
                            Provider.GOOGLE,
                            providerId
                    );
                    customer.setEnabled(true);

                    customerService.save(customer);

                    String jwt = jwtService.generateToken(customer);
                    return JWTAuthenticationResponseDTO.fromUser(customer, jwt);
                });
    }

    @Override
    public JWTAuthenticationResponseDTO registerGoogleLocalWorker(String idTokenString, String inviteToken) {
        GoogleIdToken.Payload payload = verifyGoogleIdTokenAndGetPayload(idTokenString);

        String email = payload.getEmail();
        String givenName = (String) payload.get("given_name");
        String familyName = (String) payload.get("family_name");
        String providerId = payload.getSubject();

        validateTokenEmail(inviteToken, email);

        return userRepository.findByEmail(email)
                .map(user -> {
                    String jwt = jwtService.generateToken(user);
                    return JWTAuthenticationResponseDTO.fromUser(user, jwt);
                })
                .orElseGet(() -> {
                    LocalWorker localWorker = new LocalWorker(
                            givenName,
                            familyName,
                            email,
                            passwordEncoder.encode(UUID.randomUUID().toString()),
                            null,
                            Role.ROLE_LOCAL_WORKER,
                            Provider.GOOGLE,
                            providerId
                    );
                    localWorker.setEnabled(true);

                    localWorkerService.save(localWorker);

                    String jwt = jwtService.generateToken(localWorker);
                    return JWTAuthenticationResponseDTO.fromUser(localWorker, jwt);
                });
    }

    @Override
    public JWTAuthenticationResponseDTO registerGoogleLocalManager(String idTokenString, String inviteToken) {
        GoogleIdToken.Payload payload = verifyGoogleIdTokenAndGetPayload(idTokenString);

        String email = payload.getEmail();
        String givenName = (String) payload.get("given_name");
        String familyName = (String) payload.get("family_name");
        String providerId = payload.getSubject();

        validateTokenEmail(inviteToken, email);

        return userRepository.findByEmail(email)
                .map(user -> {
                    String jwt = jwtService.generateToken(user);
                    return JWTAuthenticationResponseDTO.fromUser(user, jwt);
                })
                .orElseGet(() -> {
                    LocalManager localManager = new LocalManager(
                            givenName,
                            familyName,
                            email,
                            passwordEncoder.encode(UUID.randomUUID().toString()),
                            null,
                            Role.ROLE_LOCAL_MANAGER,
                            Provider.GOOGLE,
                            providerId
                    );
                    localManager.setEnabled(true);

                    localManagerService.save(localManager);

                    String jwt = jwtService.generateToken(localManager);
                    return JWTAuthenticationResponseDTO.fromUser(localManager, jwt);
                });
    }

    @Override
    public JWTAuthenticationResponseDTO loginWithGoogle(String idTokenString) {
        GoogleIdToken.Payload payload = verifyGoogleIdTokenAndGetPayload(idTokenString);

        String email = payload.getEmail();
        User user = userRepository.findByEmail(email).orElseThrow(() -> new EmailNotFoundException(email));

        if (!user.isEnabled()) {
            throw new IllegalStateException("Account not enabled.");
        }

        String jwt = jwtService.generateToken(user);

        userService.updateUserActivity(user.getId());

        return JWTAuthenticationResponseDTO.fromUser(user, jwt);
    }

    private GoogleIdToken.Payload verifyGoogleIdTokenAndGetPayload(String idTokenString) {
        if (idTokenString == null || idTokenString.isBlank()) {
            throw new IllegalArgumentException("idToken is required");
        }

        GoogleIdToken idToken;
        try {
            idToken = googleIdTokenVerifier.verify(idTokenString);
        } catch (GeneralSecurityException | IOException e) {
            throw new VerifyException(e);
        }

        if (idToken == null) {
            throw new IllegalArgumentException("Invalid Google ID token");
        }

        GoogleIdToken.Payload payload = idToken.getPayload();

        Boolean emailVerified = (Boolean) payload.get("email_verified");
        if (emailVerified != null && !emailVerified) {
            throw new VerifyException("Google account email not verified");
        }

        return payload;
    }

    private void validateTokenEmail(String token, String email) {
        String tokenEmail = jwtService.extractClaim(token, claims -> claims.get("email", String.class));
        if (!tokenEmail.equals(email)) {
            throw new IllegalArgumentException("Email must match the token's email");
        }
    }

    private String generateVerificationCode() {
        Random random = new Random();
        return String.valueOf(random.nextInt(900000) + 100000);
    }
}