package mk.ukim.finki.it.reservengo.web.controller;

import jakarta.mail.MessagingException;
import mk.ukim.finki.it.reservengo.dto.jwtDTO.JWTAuthenticationRequestDTO;
import mk.ukim.finki.it.reservengo.dto.jwtDTO.JWTAuthenticationResponseDTO;
import mk.ukim.finki.it.reservengo.dto.jwtDTO.JWTLoginDTO;
import mk.ukim.finki.it.reservengo.dto.jwtDTO.VerificationDTO;
import mk.ukim.finki.it.reservengo.service.intf.AuthService;
import org.springframework.http.HttpStatus;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.*;

@RestController
@RequestMapping("/api/auth")
public class AuthController {
    private final AuthService authenticationService;

    public AuthController(AuthService authenticationService) {
        this.authenticationService = authenticationService;
    }

    @PostMapping(value = "/register/customer")
    public ResponseEntity<JWTAuthenticationResponseDTO> registerCustomer(@RequestBody JWTAuthenticationRequestDTO jwtAuthenticationRequestDTO) throws MessagingException {
        return new ResponseEntity<>(authenticationService.registerCustomer(jwtAuthenticationRequestDTO), HttpStatus.CREATED);
    }

    @PostMapping("/register/local-worker")
    public ResponseEntity<JWTAuthenticationResponseDTO> registerLocalWorker(@RequestBody JWTAuthenticationRequestDTO jwtAuthenticationRequestDTO,
                                                                            @RequestHeader(value = "Invite-Token") String token) {
        return new ResponseEntity<>(authenticationService.registerLocalWorker(jwtAuthenticationRequestDTO, token), HttpStatus.CREATED);
    }

    @PostMapping("/register/local-manager")
    public ResponseEntity<JWTAuthenticationResponseDTO> registerLocalManager(@RequestBody JWTAuthenticationRequestDTO jwtAuthenticationRequestDTO,
                                                                             @RequestHeader(value = "Invite-Token") String token) {
        return new ResponseEntity<>(authenticationService.registerLocalManager(jwtAuthenticationRequestDTO, token), HttpStatus.CREATED);
    }

    @PostMapping("/login")
    public ResponseEntity<JWTAuthenticationResponseDTO> login(@RequestBody JWTLoginDTO jwtLoginDTO) throws MessagingException {
        return new ResponseEntity<>(authenticationService.login(jwtLoginDTO), HttpStatus.OK);
    }

    @PostMapping("/oauth/google/customer")
    public ResponseEntity<JWTAuthenticationResponseDTO> registerGoogleCustomer(@RequestHeader(value = "X-Google-Token") String idToken) {
        return new ResponseEntity<>(authenticationService.registerGoogleCustomer(idToken), HttpStatus.CREATED);
    }

    @PostMapping("/oauth/google/local-worker")
    public ResponseEntity<JWTAuthenticationResponseDTO> registerGoogleLocalWorker(@RequestHeader(value = "X-Google-Token") String idToken,
                                                                                  @RequestHeader(value = "Invite-Token") String inviteToken) {
        return new ResponseEntity<>(authenticationService.registerGoogleLocalWorker(idToken, inviteToken), HttpStatus.CREATED);
    }

    @PostMapping("/oauth/google/local-manager")
    public ResponseEntity<JWTAuthenticationResponseDTO> registerGoogleLocalManager(@RequestHeader(value = "X-Google-Token") String idToken,
                                                                                   @RequestHeader(value = "Invite-Token") String inviteToken) {
        return new ResponseEntity<>(authenticationService.registerGoogleLocalManager(idToken, inviteToken), HttpStatus.CREATED);
    }

    @PostMapping("/oauth/google/login")
    public ResponseEntity<JWTAuthenticationResponseDTO> loginWithGoogle(@RequestHeader(value = "X-Google-Token") String idToken) {
        return new ResponseEntity<>((authenticationService.loginWithGoogle(idToken)), HttpStatus.OK);
    }

    @PatchMapping("/enable")
    public ResponseEntity<Void> reactivateAccount(@RequestBody JWTLoginDTO jwtLoginDTO) {
        authenticationService.reactivateProfile(jwtLoginDTO);
        return new ResponseEntity<>(HttpStatus.OK);
    }

    @GetMapping("/register/invite/check")
    public ResponseEntity<Void> checkInvite() {
        return new ResponseEntity<>(HttpStatus.OK);
    }

    @PatchMapping("/verify")
    public ResponseEntity<Void> verifyAccount(@RequestBody VerificationDTO verificationDTO) throws MessagingException {
        authenticationService.verifyCustomer(verificationDTO.email(), verificationDTO.verificationCode());
        return new ResponseEntity<>(HttpStatus.OK);
    }

    @PostMapping("/resend-verification")
    public ResponseEntity<Void> resendVerificationCode(@RequestBody VerificationDTO verificationDTO) throws MessagingException {
        authenticationService.resendVerificationCode(verificationDTO.email());
        return new ResponseEntity<>(HttpStatus.OK);
    }
}