package mk.ukim.finki.it.reservengo.model.domain;

import jakarta.persistence.*;
import lombok.AllArgsConstructor;
import lombok.Data;
import lombok.EqualsAndHashCode;
import lombok.NoArgsConstructor;
import mk.ukim.finki.it.reservengo.config.listeners.Auditable;
import mk.ukim.finki.it.reservengo.model.enumerations.LocalType;
import mk.ukim.finki.it.reservengo.model.enumerations.ProvidedService;
import org.springframework.data.jpa.domain.support.AuditingEntityListener;

import java.time.DayOfWeek;
import java.time.LocalDateTime;
import java.time.LocalTime;
import java.util.List;
import java.util.Map;


@EqualsAndHashCode(callSuper = true)
@Entity
@Table(name = "local_")
@Data
@AllArgsConstructor
@NoArgsConstructor
@EntityListeners(AuditingEntityListener.class)
public class Local extends Auditable {
    @Id
    @GeneratedValue(strategy = GenerationType.IDENTITY)
    private Long id;

    private String name;

    private String description;

    private String address;

    @ElementCollection
    private List<WorkingHour> workingHours;

    @Enumerated(EnumType.STRING)
    @ElementCollection
    private List<ProvidedService> availableServices;

    @Enumerated(EnumType.STRING)
    private LocalType localType;

    @ElementCollection
    @MapKeyColumn(name = "customer_id")
    private Map<Long, Integer> ratings;

    @OneToMany(mappedBy = "local", cascade = CascadeType.ALL, orphanRemoval = true)
    private List<Event> events;

    @ElementCollection
    private List<String> photos;

    private String menuLink;

    @Embedded
    private Contact contact;

    @OneToMany(mappedBy = "local", cascade = CascadeType.ALL)
    private List<Reservation> reservations;

    private String logoUrl;

    public Local(String name, String description, String address, List<WorkingHour> workingHours, List<ProvidedService> availableServices, Map<Long, Integer> ratings, List<Event> events, String logoUrl, List<String> photos, String menuLink, Contact contact, List<Reservation> reservations, LocalType localType) {
        this.name = name;
        this.description = description;
        this.address = address;
        this.workingHours = workingHours;
        this.availableServices = availableServices;
        this.ratings = ratings;
        this.events = events;
        this.logoUrl = logoUrl;
        this.photos = photos;
        this.menuLink = menuLink;
        this.contact = contact;
        this.reservations = reservations;
        this.localType = localType;
    }

    public Local(String name) {
        this.name = name;
    }

    public double getAverageRating() {
        if (ratings == null || ratings.isEmpty()) {
            return 0;
        }

        return ratings.values().stream()
                .mapToInt(Integer::intValue)
                .average()
                .orElse(0.0);
    }

    public boolean isOpenNow() {
        if (workingHours == null || workingHours.isEmpty()) {
            return false;
        }

        LocalDateTime now = LocalDateTime.now();
        DayOfWeek today = now.getDayOfWeek();
        LocalTime currentTime = now.toLocalTime();

        return workingHours.stream()
                .filter(wh -> wh.getDayOfWeek().name().equals(today.name()))
                .anyMatch(wh ->
                        currentTime.isAfter(wh.getOpenTime()) && currentTime.isBefore(wh.getCloseTime())
                );
    }
}