package mk.ukim.finki.it.reservengo.service.impl;

import mk.ukim.finki.it.reservengo.dto.workerDTO.DisplayLocalReservationDTO;
import mk.ukim.finki.it.reservengo.model.domain.Local;
import mk.ukim.finki.it.reservengo.model.domain.LocalWorker;
import mk.ukim.finki.it.reservengo.model.domain.Reservation;
import mk.ukim.finki.it.reservengo.model.enumerations.Position;
import mk.ukim.finki.it.reservengo.model.enumerations.ReservationStatus;
import mk.ukim.finki.it.reservengo.model.enumerations.Role;
import mk.ukim.finki.it.reservengo.model.exceptions.*;
import mk.ukim.finki.it.reservengo.repository.LocalRepository;
import mk.ukim.finki.it.reservengo.repository.LocalWorkerRepository;
import mk.ukim.finki.it.reservengo.repository.ReservationRepository;
import mk.ukim.finki.it.reservengo.service.intf.LocalWorkerService;
import mk.ukim.finki.it.reservengo.service.intf.UserService;
import org.springframework.stereotype.Service;

import java.util.List;
import java.util.stream.Collectors;

@Service
public class LocalWorkerServiceImpl implements LocalWorkerService {
    private final LocalWorkerRepository localWorkerRepository;
    private final UserService userService;
    private final LocalRepository localRepository;
    private final ReservationRepository reservationRepository;

    public LocalWorkerServiceImpl(LocalWorkerRepository localWorkerRepository, UserService userService, LocalRepository localRepository, ReservationRepository reservationRepository) {
        this.localWorkerRepository = localWorkerRepository;
        this.userService = userService;
        this.localRepository = localRepository;
        this.reservationRepository = reservationRepository;
    }

    @Override
    public void save(LocalWorker localWorker) {
        if (userService.emailExists(localWorker.getEmail())) {
            throw new UserEmailAlreadyExistsException(localWorker.getEmail());
        }
        localWorkerRepository.save(localWorker);
    }

    @Override
    public LocalWorker findWorkerById(Long id) {
        return localWorkerRepository.findById(id).orElseThrow(() -> new WorkerIdNotFoundException(id));
    }

    @Override
    public void changePosition(LocalWorker worker, Position position) {
        worker.setPosition(position);
        localWorkerRepository.save(worker);
    }

    @Override
    public List<LocalWorker> findAllByLocalId(Long id) {
        return localWorkerRepository.findByLocal_Id(id);
    }

    @Override
    public List<LocalWorker> findAllUnassigned() {
        return localWorkerRepository.findAll().stream()
                .filter(worker -> worker.getLocal() == null)
                .filter(worker -> worker.getPosition() == null)
                .collect(Collectors.toList());
    }

    @Override
    public void updateLocalAssignment(LocalWorker worker, Local local) {
        worker.setLocal(local);
        if (local == null) {
            worker.setPosition(null);
        }
        localWorkerRepository.save(worker);
    }

    @Override
    public List<DisplayLocalReservationDTO> viewLocalReservations(Long id,ReservationStatus status) {
        Local local = localRepository.findById(id).orElseThrow(() -> new LocalIdNotFoundException(id));

        List<Reservation> reservations;

        if (status != null){
            reservations = reservationRepository.findByLocalIdAndStatus(local.getId(),status);
        } else {
            reservations = reservationRepository.findByLocalId(local.getId());
        }
        return DisplayLocalReservationDTO.from(reservations);
    }

    @Override
    public void validateWorker(Long id, Long localId) {
        LocalWorker worker = localWorkerRepository.findById(id).orElseThrow(() -> new WorkerIdNotFoundException(id));

        if (worker.getUserRole() != Role.ROLE_LOCAL_WORKER) {
            throw new UnauthorizedException(worker.getId());
        }

        if (!worker.getLocal().getId().equals(localId)) {
            throw new UnauthorizedException(worker.getId(), localId);
        }
    }

    @Override
    public void acceptReservation(Long id, Long reservationId) {
        Reservation reservation = reservationRepository.findById(reservationId).orElseThrow(() -> new ReservationIdNotFoundException(reservationId));
        Long localId = reservation.getLocal().getId();

        validateWorker(id, localId);

        reservation.setStatus(ReservationStatus.ACCEPTED);
        reservationRepository.save(reservation);
    }

    @Override
    public void denyReservation(Long id, Long reservationId) {
        Reservation reservation = reservationRepository.findById(reservationId).orElseThrow(() -> new ReservationIdNotFoundException(reservationId));
        Long localId = reservation.getLocal().getId();

        validateWorker(id, localId);

        reservation.setStatus(ReservationStatus.CANCELED);
        reservationRepository.save(reservation);
    }

    @Override
    public void finishReservation(Long id, Long reservationId) {
        Reservation reservation = reservationRepository.findById(reservationId).orElseThrow(() -> new ReservationIdNotFoundException(reservationId));
        Long localId = reservation.getLocal().getId();

        validateWorker(id, localId);

        reservation.setStatus(ReservationStatus.FINISHED);
        reservationRepository.save(reservation);
    }
}
