package mk.ukim.finki.it.reservengo.service.impl;

import mk.ukim.finki.it.reservengo.dto.eventDTO.CreateEventDTO;
import mk.ukim.finki.it.reservengo.dto.localDTO.CreateLocalDetailsDTO;
import mk.ukim.finki.it.reservengo.dto.localDTO.DeleteLocalPhotosDTO;
import mk.ukim.finki.it.reservengo.dto.localDTO.DeleteLocalPhotosResultDTO;
import mk.ukim.finki.it.reservengo.dto.userDTO.EditUserEmailDTO;
import mk.ukim.finki.it.reservengo.dto.workerDTO.PositionDTO;
import mk.ukim.finki.it.reservengo.model.domain.Event;
import mk.ukim.finki.it.reservengo.model.domain.Local;
import mk.ukim.finki.it.reservengo.model.domain.LocalManager;
import mk.ukim.finki.it.reservengo.model.domain.LocalWorker;
import mk.ukim.finki.it.reservengo.model.enumerations.Position;
import mk.ukim.finki.it.reservengo.model.enumerations.Role;
import mk.ukim.finki.it.reservengo.model.exceptions.*;
import mk.ukim.finki.it.reservengo.repository.LocalManagerRepository;
import mk.ukim.finki.it.reservengo.service.intf.*;
import org.springframework.security.access.AccessDeniedException;
import org.springframework.stereotype.Service;
import org.springframework.web.multipart.MultipartFile;

import java.util.List;
import java.util.stream.Collectors;

@Service
public class LocalManagerServiceImpl implements LocalManagerService {

    private final LocalService localService;
    private final LocalManagerRepository localManagerRepository;
    private final UserService userService;
    private final EventService eventService;
    private final LocalWorkerService workerService;
    private final JWTService jwtService;
    private final EmailService emailService;

    public LocalManagerServiceImpl(LocalService localService, LocalManagerRepository localManagerRepository, UserService userService, EventService eventService, LocalWorkerService workerService, JWTService jwtService, EmailService emailService) {
        this.localService = localService;
        this.localManagerRepository = localManagerRepository;
        this.userService = userService;
        this.eventService = eventService;
        this.workerService = workerService;
        this.jwtService = jwtService;
        this.emailService = emailService;
    }

    @Override
    public void save(LocalManager localManager) {
        if (userService.emailExists(localManager.getEmail())) {
            throw new UserEmailAlreadyExistsException(localManager.getEmail());
        }
        localManagerRepository.save(localManager);
    }

    public List<LocalManager> findAllByLocalId(Long localId) {
        return localManagerRepository.findByLocal_Id(localId);
    }

    @Override
    public List<LocalManager> findAll() {
        return localManagerRepository.findAll();
    }

    @Override
    public LocalManager findManagerById(Long id) {
        return localManagerRepository.findById(id).orElseThrow(() -> new LocalManagerIdNotFoundException(id));
    }

    @Override
    public List<LocalManager> findAllUnassigned() {
        return localManagerRepository.findAll().stream()
                .filter(manager -> manager.getLocal() == null)
                .filter(manager -> manager.getPosition() == null)
                .collect(Collectors.toList());
    }

    @Override
    public void updateLocalAssignment(LocalManager manager, Local local, Position position) {
        manager.setLocal(local);
        manager.setPosition(position);

        localManagerRepository.save(manager);
    }

    @Override
    public Local findLocalByManagerId(Long id) {
        LocalManager localManager = localManagerRepository.findById(id).orElseThrow(() -> new LocalManagerIdNotFoundException(id));
        if (localManager.getLocal() == null) {
            throw new ManagerNotAssignedException(id);
        }
        return localManager.getLocal();
    }

    @Override
    public Local editLocal(Long managerId, CreateLocalDetailsDTO createLocalDetailsDTO) {
        LocalManager manager = localManagerRepository.findById(managerId).orElseThrow(() -> new LocalManagerIdNotFoundException(managerId));
        if (manager.getLocal() == null) {
            throw new ManagerNotAssignedException(manager.getId());
        }
        return localService.edit(manager.getLocal().getId(), createLocalDetailsDTO);
    }

    @Override
    public String uploadLocalLogo(Long managerId, MultipartFile logoFile) {
        Local local = findLocalByManagerId(managerId);

        return localService.addLogo(local.getId(), logoFile);
    }

    @Override
    public void deleteLocalLogo(Long managerId) {
        Local local = findLocalByManagerId(managerId);

        localService.deleteLogo(local.getId());
    }

    @Override
    public String uploadLocalPhoto(Long managerId, MultipartFile photoFile) {
        Local local = findLocalByManagerId(managerId);

        return localService.addPhoto(local.getId(), photoFile);
    }

    @Override
    public DeleteLocalPhotosResultDTO deleteLocalPhoto(Long managerId, DeleteLocalPhotosDTO deleteLocalPhotosDTO) {
        Local local = findLocalByManagerId(managerId);

        return localService.deletePhotos(local.getId(), deleteLocalPhotosDTO.localPhotosUrls());
    }

    @Override
    public void addEvent(Long managerId, CreateEventDTO createEventDTO) {
        Local local = findLocalByManagerId(managerId);
        Event event = createEventDTO.toEvent(local);

        localService.addEvent(local, event);
    }

    @Override
    public void deleteEvent(Long managerId, Long eventId) {
        Local local = findLocalByManagerId(managerId);
        Event event = eventService.findEventById(eventId);

        if (!event.getLocal().getId().equals(local.getId())) {
            throw new EventNotInLocalException(eventId);
        }

        localService.deleteEvent(local, event);
    }

    @Override
    public Event editEvent(Long managerId, Long eventId, CreateEventDTO createEventDTO) {
        Local local = findLocalByManagerId(managerId);
        Event event = eventService.findEventById(eventId);

        if (!event.getLocal().getId().equals(local.getId())) {
            throw new EventNotInLocalException(eventId);
        }

        return localService.editEvent(local, event, createEventDTO);
    }

    @Override
    public List<LocalWorker> findWorkersByLocal(Local local) {
        return workerService.findAllByLocalId(local.getId());
    }

    @Override
    public List<LocalWorker> findUnassignedWorkers() {
        return workerService.findAllUnassigned();
    }

    @Override
    public void assignWorkerToLocal(Local local, Long workerId) {
        LocalWorker worker = workerService.findWorkerById(workerId);
        if (worker.getLocal() != null) {
            throw new WorkerAlreadyAssignedException(worker.getId(), worker.getLocal().getId());
        }
        workerService.updateLocalAssignment(worker, local);
    }

    @Override
    public void removeWorkerFromLocal(Long managerId, Long workerId) {
        LocalWorker worker = workerService.findWorkerById(workerId);
        if (worker.getLocal() == null) {
            throw new WorkerNotAssignedException(workerId);
        }
        LocalManager manager = findManagerById(managerId);
        if (!worker.getLocal().getId().equals(manager.getLocal().getId())) {
            throw new AccessDeniedException("You are not allowed to remove a worker that is not assigned to your local!");
        }
        workerService.updateLocalAssignment(worker, null);
    }

    @Override
    public void changePosition(Long managerId, Long workerId, PositionDTO positionDTO) {
        LocalWorker worker = workerService.findWorkerById(workerId);
        if (worker.getLocal() == null) {
            throw new WorkerNotAssignedException(workerId);
        }
        LocalManager manager = findManagerById(managerId);
        if (!worker.getLocal().getId().equals(manager.getLocal().getId())) {
            throw new AccessDeniedException("You are not allowed to change the position of a worker that is not assigned to your local!");
        }
        workerService.changePosition(worker, positionDTO.position());
    }

    @Override
    public void inviteWorker(Long managerId, EditUserEmailDTO emailDTO) {
        LocalManager manager = localManagerRepository.findById(managerId).orElseThrow(() -> new LocalManagerIdNotFoundException(managerId));
        if (manager.getLocal() == null) {
            throw new ManagerNotAssignedException(managerId);
        }

        String token = jwtService.generateInviteToken(emailDTO.newEmail(), Role.ROLE_LOCAL_WORKER);
        emailService.sendWorkerInvitationEmail(emailDTO.newEmail(), token, manager.getEmail(), manager.getLocal().getName());
    }
}
