import { config } from '@/constants/Api_config'
import { HttpClient } from '@/Api_Classes/HttpClient'

const BASE_API_URL = config.API_BASE_URL

export type CreateReservationDTO = {
  timeOfReservation: string
  capacity: number
  description?: string
}

export type ReservationStatus = 'PENDING' | 'CONFIRMED' | 'CANCELLED' | string

export type DisplayReservationDTO = {
  id?: number | string
  localName: string
  localLogo: string | null
  timeOfReservation: string
  capacity: number
  description?: string
  status: ReservationStatus
}

export type EditReservationDTO = {
  timeOfReservation: string // ISO string
  capacity: number
  description: string
}

export type DeleteReservationDTO = {
  reservationIds: Array<number | string>
}

class ReservationsRepository {
  private readonly httpClient: HttpClient

  constructor(baseUrl: string) {
    this.httpClient = new HttpClient(baseUrl)
  }

  // POST /api/reservations/make-reservation
  makeReservation(dto: CreateReservationDTO, localId: number): Promise<any> {
    return this.httpClient.post(`${localId}/create`, dto)
  }

  // GET /api/reservations
  listMyReservations(): Promise<DisplayReservationDTO[]> {
    return this.httpClient.get('')
  }

  // PUT /api/reservations/{id}/cancel
  cancelReservation(reservationId: number | string): Promise<void> {
    return this.httpClient.put(`${reservationId}/cancel`, null)
  }

  // GET /api/reservations/{id}
  getReservationDetails(reservationId: number | string): Promise<DisplayReservationDTO> {
    return this.httpClient.get(`${reservationId}`)
  }

  // PUT /api/reservations/{id}/edit
  editReservation(
    reservationId: number | string,
    dto: EditReservationDTO,
  ): Promise<DisplayReservationDTO> {
    return this.httpClient.put(`${reservationId}/edit`, dto)
  }

  // DELETE /api/customer/reservations/delete
  deleteReservations(dto: DeleteReservationDTO): Promise<void> {
    return this.httpClient.delete('delete', dto)
  }
}

export const useReservations = new ReservationsRepository(
  BASE_API_URL + '/api/customer/reservations',
)
