package mk.ukim.finki.it.reservengo.service.impl;

import mk.ukim.finki.it.reservengo.dto.adminDTO.DisplayAdminLocalsDTO;
import mk.ukim.finki.it.reservengo.dto.localDTO.CreateLocalDTO;
import mk.ukim.finki.it.reservengo.dto.userDTO.DisplayUserDTO;
import mk.ukim.finki.it.reservengo.dto.userDTO.EditUserEmailDTO;
import mk.ukim.finki.it.reservengo.model.domain.Local;
import mk.ukim.finki.it.reservengo.model.domain.LocalManager;
import mk.ukim.finki.it.reservengo.model.enumerations.Position;
import mk.ukim.finki.it.reservengo.model.enumerations.Role;
import mk.ukim.finki.it.reservengo.model.exceptions.ManagerAlreadyAssignedException;
import mk.ukim.finki.it.reservengo.model.exceptions.ManagerNotAssignedException;
import mk.ukim.finki.it.reservengo.service.intf.*;
import org.springframework.stereotype.Service;

import java.util.List;

@Service
public class AdminServiceImpl implements AdminService {

    private final LocalService localService;
    private final LocalManagerService localManagerService;
    private final EmailService emailService;
    private final JWTService jwtService;

    public AdminServiceImpl(LocalService localService, LocalManagerService localManagerService, EmailService emailService, JWTService jwtService) {
        this.localService = localService;
        this.localManagerService = localManagerService;
        this.emailService = emailService;
        this.jwtService = jwtService;
    }

    @Override
    public void addLocal(CreateLocalDTO localDTO) {
        localService.save(localDTO.name());
    }

    @Override
    public void deleteLocal(Long id) {
        for (LocalManager manager : localManagerService.findAllByLocalId(id)) {
            removeManagerFromLocal(manager.getId());
        }
        localService.delete(id);
    }

    @Override
    public List<DisplayAdminLocalsDTO> getLocals() {
        return DisplayAdminLocalsDTO.fromLocals(localService.listAll());
    }

    @Override
    public List<DisplayUserDTO> findManagersByLocalId(Long localId) {
        List<LocalManager> managers = localManagerService.findAllByLocalId(localId);
        return DisplayUserDTO.fromUsers(managers);
    }

    @Override
    public List<DisplayUserDTO> findUnassignedManagers() {
        List<LocalManager> managers = localManagerService.findAllUnassigned();
        return DisplayUserDTO.fromUsers(managers);
    }

    @Override
    public void assignManagerToLocal(Long localId, Long managerId) {
        LocalManager manager = localManagerService.findManagerById(managerId);
        if (manager.getLocal() != null) {
            throw new ManagerAlreadyAssignedException(manager.getId(), manager.getLocal().getId());
        }
        Local local = localService.findLocalById(localId);
        localManagerService.updateLocalAssignment(manager, local, Position.MANAGER);
    }

    @Override
    public void removeManagerFromLocal(Long managerId) {
        LocalManager manager = localManagerService.findManagerById(managerId);
        if (manager.getLocal() == null) {
            throw new ManagerNotAssignedException(managerId);
        }
        localManagerService.updateLocalAssignment(manager, null, null);
    }

    @Override
    public void inviteManager(EditUserEmailDTO emailDTO) {
        String token = jwtService.generateInviteToken(emailDTO.newEmail(), Role.ROLE_LOCAL_MANAGER);
        emailService.sendManagerInvitationEmail(emailDTO.newEmail(), token);
    }
}
