package mk.ukim.finki.it.reservengo.service.impl;

import mk.ukim.finki.it.reservengo.dto.localDTO.CreateLocalDetailsDTO;
import mk.ukim.finki.it.reservengo.model.domain.Local;
import mk.ukim.finki.it.reservengo.model.domain.LocalManager;
import mk.ukim.finki.it.reservengo.model.exceptions.LocalManagerIdNotFoundException;
import mk.ukim.finki.it.reservengo.model.exceptions.UserEmailAlreadyExistsException;
import mk.ukim.finki.it.reservengo.repository.LocalManagerRepository;
import mk.ukim.finki.it.reservengo.service.intf.LocalManagerService;
import mk.ukim.finki.it.reservengo.service.intf.LocalService;
import mk.ukim.finki.it.reservengo.service.intf.UserService;
import org.springframework.stereotype.Service;
import org.springframework.web.multipart.MultipartFile;

import java.util.List;

@Service
public class LocalManagerServiceImpl implements LocalManagerService {

    private final LocalService localService;
    private final LocalManagerRepository localManagerRepository;
    private final UserService userService;

    public LocalManagerServiceImpl(LocalService localService, LocalManagerRepository localManagerRepository, UserService userService) {
        this.localService = localService;
        this.localManagerRepository = localManagerRepository;
        this.userService = userService;
    }

    @Override
    public void uploadLocalLogo(Long localId, MultipartFile logoFile) {
        localService.addLogo(localId, logoFile);
    }

    @Override
    public void save(LocalManager localManager) {
        if (userService.emailExists(localManager.getEmail())) {
            throw new UserEmailAlreadyExistsException(localManager.getEmail());
        }
        localManagerRepository.save(localManager);
    }

    @Override
    public void deleteLocalLogo(Long localId) {
        localService.deleteLogo(localId);
    }

    @Override
    public void uploadLocalPhotos(Long localId, List<MultipartFile> photos) {
        localService.uploadPhotos(localId, photos);
    }

    @Override
    public void deleteLocalPhotos(Long localId, List<String> photoPaths) {
        localService.deletePhotos(localId, photoPaths);
    }

    @Override
    public void uploadLocalMenu(Long localId, MultipartFile localMenuFile) {
        localService.uploadMenu(localId, localMenuFile);
    }

    @Override
    public void deleteLocalMenuPhoto(Long localId) {
        localService.deleteMenu(localId);
    }

    public List<LocalManager> findAllByLocalId(Long localId) {
        return localManagerRepository.findByLocal_Id(localId);
    }

    @Override
    public List<LocalManager> findAll() {
        return localManagerRepository.findAll();
    }

    @Override
    public LocalManager findManagerById(Long id) {
        return localManagerRepository.findById(id).orElseThrow(() -> new LocalManagerIdNotFoundException(id));
    }

    @Override
    public void update(Long id, LocalManager manager) {
        LocalManager existingManager = localManagerRepository.findById(id).orElseThrow(() -> new LocalManagerIdNotFoundException(id));

        existingManager.setLocal(manager.getLocal());
        existingManager.setPosition(manager.getPosition());

        localManagerRepository.save(existingManager);
    }

    @Override
    public Local findLocalByManagerId(Long id) {
        LocalManager localManager = localManagerRepository.findById(id).orElseThrow(() -> new LocalManagerIdNotFoundException(id));
        return localManager.getLocal();
    }

    @Override
    public void editLocalInfo(Long localId, CreateLocalDetailsDTO createLocalDetailsDTO) {
        localService.updateInfoLocal(localId, createLocalDetailsDTO);
    }
}
