package mk.ukim.finki.it.reservengo.service.impl;

import mk.ukim.finki.it.reservengo.dto.localDTO.CreateLocalDetailsDTO;
import mk.ukim.finki.it.reservengo.model.domain.Local;
import mk.ukim.finki.it.reservengo.model.domain.LocalManager;
import mk.ukim.finki.it.reservengo.model.enumerations.Position;
import mk.ukim.finki.it.reservengo.model.exceptions.LocalManagerIdNotFoundException;
import mk.ukim.finki.it.reservengo.model.exceptions.ManagerNotAssignedException;
import mk.ukim.finki.it.reservengo.model.exceptions.UserEmailAlreadyExistsException;
import mk.ukim.finki.it.reservengo.repository.LocalManagerRepository;
import mk.ukim.finki.it.reservengo.service.intf.LocalManagerService;
import mk.ukim.finki.it.reservengo.service.intf.LocalService;
import mk.ukim.finki.it.reservengo.service.intf.UserService;
import org.springframework.stereotype.Service;
import org.springframework.web.multipart.MultipartFile;

import java.util.List;
import java.util.stream.Collectors;

@Service
public class LocalManagerServiceImpl implements LocalManagerService {

    private final LocalService localService;
    private final LocalManagerRepository localManagerRepository;
    private final UserService userService;

    public LocalManagerServiceImpl(LocalService localService, LocalManagerRepository localManagerRepository, UserService userService) {
        this.localService = localService;
        this.localManagerRepository = localManagerRepository;
        this.userService = userService;
    }

    @Override
    public void save(LocalManager localManager) {
        if (userService.emailExists(localManager.getEmail())) {
            throw new UserEmailAlreadyExistsException(localManager.getEmail());
        }
        localManagerRepository.save(localManager);
    }

    public List<LocalManager> findAllByLocalId(Long localId) {
        return localManagerRepository.findByLocal_Id(localId);
    }

    @Override
    public List<LocalManager> findAll() {
        return localManagerRepository.findAll();
    }

    @Override
    public LocalManager findManagerById(Long id) {
        return localManagerRepository.findById(id).orElseThrow(() -> new LocalManagerIdNotFoundException(id));
    }

    @Override
    public List<LocalManager> findAllUnassigned() {
        return localManagerRepository.findAll().stream()
                .filter(manager -> manager.getLocal() == null)
                .filter(manager -> manager.getPosition() == null)
                .collect(Collectors.toList());
    }

    @Override
    public void updateLocalAssignment(LocalManager manager, Local local, Position position) {
        manager.setLocal(local);
        manager.setPosition(position);

        localManagerRepository.save(manager);
    }

    @Override
    public Local findLocalByManagerId(Long id) {
        LocalManager localManager = localManagerRepository.findById(id).orElseThrow(() -> new LocalManagerIdNotFoundException(id));
        if (localManager.getLocal() == null) {
            throw new ManagerNotAssignedException(id);
        }
        return localManager.getLocal();
    }

    @Override
    public Local editLocal(Long managerId, CreateLocalDetailsDTO createLocalDetailsDTO) {
        LocalManager localManager = localManagerRepository.findById(managerId).orElseThrow(() -> new LocalManagerIdNotFoundException(managerId));

        return localService.edit(localManager.getLocal().getId(), createLocalDetailsDTO);
    }

    @Override
    public String uploadLocalLogo(Long managerId, MultipartFile logoFile) {
        Local local = findLocalByManagerId(managerId);

        return localService.addLogo(local.getId(), logoFile);
    }

    @Override
    public void deleteLocalLogo(Long managerId) {
        LocalManager localManager = localManagerRepository.findById(managerId).orElseThrow(() -> new LocalManagerIdNotFoundException(managerId));

        localService.deleteLogo(localManager.getLocal().getId());
    }

    @Override
    public void uploadLocalPhotos(Long localId, List<MultipartFile> photos) {
        localService.uploadPhotos(localId, photos);
    }

    @Override
    public void deleteLocalPhotos(Long localId, List<String> photoPaths) {
        localService.deletePhotos(localId, photoPaths);
    }
}
