package mk.ukim.finki.it.reservengo.service.impl;

import mk.ukim.finki.it.reservengo.dto.localDTO.CreateLocalDetailsDTO;
import mk.ukim.finki.it.reservengo.model.domain.Local;
import mk.ukim.finki.it.reservengo.model.exceptions.*;
import mk.ukim.finki.it.reservengo.repository.LocalRepository;
import mk.ukim.finki.it.reservengo.service.intf.FileStorageService;
import mk.ukim.finki.it.reservengo.service.intf.LocalService;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.web.multipart.MultipartFile;

import java.util.ArrayList;
import java.util.List;

@Service
@Transactional
public class LocalServiceImpl implements LocalService {

    private final LocalRepository localRepository;
    private final FileStorageService fileStorageService;

    public LocalServiceImpl(LocalRepository localRepository, FileStorageService fileStorageService) {
        this.localRepository = localRepository;
        this.fileStorageService = fileStorageService;
    }

    @Override
    public List<Local> listAll() {
        return localRepository.findAll();
    }

    @Override
    public Local findLocalById(Long id) {
        return localRepository.findById(id).orElseThrow(() -> new LocalIdNotFoundException(id));
    }

    @Override
    public void save(String name) {
        Local local = new Local(name);
        localRepository.save(local);
    }

    @Override
    public Local edit(Long localId, CreateLocalDetailsDTO createLocalDetailsDTO) {
        Local local = localRepository.findById(localId).orElseThrow(() -> new LocalIdNotFoundException(localId));

        local.setName(createLocalDetailsDTO.name());
        local.setDescription(createLocalDetailsDTO.description());
        local.setAddress(createLocalDetailsDTO.address());
        local.setWorkingHours(createLocalDetailsDTO.workingHours());
        local.setAvailableServices(createLocalDetailsDTO.services());
        local.setMenuLink(createLocalDetailsDTO.menuLink());
        local.setContact(createLocalDetailsDTO.contact().toContact());

        localRepository.save(local);

        return local;
    }

    @Override
    public void delete(Long id) {
        Local local = localRepository.findById(id).orElseThrow(() -> new LocalIdNotFoundException(id));
        localRepository.delete(local);
    }

    @Override
    public String addLogo(Long localId, MultipartFile multipartFile) {
        Local local = localRepository.findById(localId).orElseThrow(() -> new LocalIdNotFoundException(localId));

        if (local.getLogoUrl() != null && !local.getLogoUrl().isEmpty()) {
            fileStorageService.deletePhotoFile(local.getLogoUrl());
        }

        String logoPath = fileStorageService.saveLogoFile(multipartFile);
        local.setLogoUrl(logoPath);
        localRepository.save(local);

        return logoPath;
    }

    @Override
    public void deleteLogo(Long localId) {
        Local local = localRepository.findById(localId).orElseThrow(() -> new LocalIdNotFoundException(localId));
        String logoUrl = local.getLogoUrl();

        if (logoUrl != null && !logoUrl.isEmpty()) {
            fileStorageService.deletePhotoFile(logoUrl);
            local.setLogoUrl(null);
            localRepository.save(local);
        } else {
            throw new LogoDeletionException("No logo found for local with id: " + localId);
        }
    }

    @Override
    public void uploadPhotos(Long localId, List<MultipartFile> photos) {
        Local local = localRepository.findById(localId).orElseThrow(() -> new LocalIdNotFoundException(localId));

        List<String> photoPaths = local.getLocalPhotos();

        if (photoPaths == null) {
            photoPaths = new ArrayList<>();
        }

        for (MultipartFile photo : photos) {
            if (!photo.isEmpty()) {
                String filePath = fileStorageService.saveLocalPhoto(localId, photo);
                photoPaths.add(filePath);
            }
        }
        local.setLocalPhotos(photoPaths);
        localRepository.save(local);
    }

    @Override
    public void deletePhotos(Long localId, List<String> photoPaths) {
        Local local = localRepository.findById(localId).orElseThrow(() -> new LocalIdNotFoundException(localId));

        List<String> uploadedPhotos = local.getLocalPhotos();

        if (uploadedPhotos == null) {
            uploadedPhotos = new ArrayList<>();
        }

        for (String photoPath : photoPaths) {
            if (uploadedPhotos.contains(photoPath)) {
                try {
                    fileStorageService.deletePhotoFile(photoPath);
                    uploadedPhotos.remove(photoPath);
                } catch (RuntimeException e) {
                    throw new PhotoDeletionException("Failed to delete photo: " + photoPath, e);
                }
            }
        }

        local.setLocalPhotos(uploadedPhotos);
        localRepository.save(local);
    }
}
