package mk.ukim.finki.it.reservengo.service.impl;

import mk.ukim.finki.it.reservengo.dto.customerDTO.RatingDTO;
import mk.ukim.finki.it.reservengo.dto.eventDTO.DisplayEventDTO;
import mk.ukim.finki.it.reservengo.dto.localDTO.DisplayLocalDTO;
import mk.ukim.finki.it.reservengo.model.domain.Customer;
import mk.ukim.finki.it.reservengo.model.domain.Event;
import mk.ukim.finki.it.reservengo.model.domain.Local;
import mk.ukim.finki.it.reservengo.model.exceptions.*;
import mk.ukim.finki.it.reservengo.repository.CustomerRepository;
import mk.ukim.finki.it.reservengo.service.intf.CustomerService;
import mk.ukim.finki.it.reservengo.service.intf.EventService;
import mk.ukim.finki.it.reservengo.service.intf.LocalService;
import mk.ukim.finki.it.reservengo.service.intf.UserService;
import org.springframework.stereotype.Service;

import java.util.List;

@Service
public class CustomerServiceImpl implements CustomerService {
    private final CustomerRepository customerRepository;
    private final LocalService localService;
    private final UserService userService;
    private final EventService eventService;

    public CustomerServiceImpl(CustomerRepository customerRepository, LocalService localService, UserService userService, EventService eventService) {
        this.customerRepository = customerRepository;
        this.localService = localService;
        this.userService = userService;
        this.eventService = eventService;
    }

    @Override
    public List<DisplayLocalDTO> listFavouriteLocals(Long id) {
        Customer customer = customerRepository.findById(id).orElseThrow(() -> new CustomerIdNotFoundException(id));
        return DisplayLocalDTO.fromLocals(customer.getFavouriteLocals());
    }

    @Override
    public void addFavouriteLocal(Long userId, Long localId) {
        Customer customer = customerRepository.findById(userId).orElseThrow(() -> new CustomerIdNotFoundException(userId));
        Local local = localService.findLocalById(localId);

        if (customer.getFavouriteLocals().contains(local)) {
            throw new LocalAlreadyFavouredException(localId);
        }

        customer.getFavouriteLocals().add(local);
        customerRepository.save(customer);
    }

    @Override
    public void removeFavouriteLocal(Long userId, Long localId) {
        Customer customer = customerRepository.findById(userId).orElseThrow(() -> new CustomerIdNotFoundException(userId));
        Local local = localService.findLocalById(localId);

        if (!customer.getFavouriteLocals().contains(local)) {
            throw new LocalAlreadyUnfavouredException(localId);
        }

        customer.getFavouriteLocals().remove(local);
        customerRepository.save(customer);
    }

    @Override
    public void save(Customer customer) {
        if (userService.emailExists(customer.getEmail())) {
            throw new UserEmailAlreadyExistsException(customer.getEmail());
        }
        customerRepository.save(customer);
    }

    @Override
    public List<DisplayEventDTO> listFavouriteEvents(Long id) {
        Customer customer = customerRepository.findById(id).orElseThrow(() -> new CustomerIdNotFoundException(id));
        return DisplayEventDTO.fromEvents(customer.getFavouriteEvents());
    }

    @Override
    public void addFavouriteEvent(Long userId, Long eventId) {
        Customer customer = customerRepository.findById(userId).orElseThrow(() -> new CustomerIdNotFoundException(userId));
        Event event = eventService.findEventById(eventId);

        if (customer.getFavouriteEvents().contains(event)) {
            throw new EventAlreadyFavouredException(eventId);
        }

        customer.getFavouriteEvents().add(event);
        customerRepository.save(customer);
    }

    @Override
    public void removeFavouriteEvent(Long userId, Long eventId) {
        Customer customer = customerRepository.findById(userId).orElseThrow(() -> new CustomerIdNotFoundException(userId));
        Event event = eventService.findEventById(eventId);

        if (!customer.getFavouriteEvents().contains(event)) {
            throw new EventAlreadyUnfavouredException(eventId);
        }

        customer.getFavouriteEvents().remove(event);
        customerRepository.save(customer);
    }

    @Override
    public void rateLocal(Long customerId, Long localId, RatingDTO ratingDTO) {
        customerRepository.findById(customerId).orElseThrow(() -> new CustomerIdNotFoundException(customerId));
        localService.addRating(customerId, localId, ratingDTO.rating());
    }

    @Override
    public void removeRating(Long customerId, Long localId) {
        customerRepository.findById(customerId).orElseThrow(() -> new CustomerIdNotFoundException(customerId));
        localService.removeRating(customerId, localId);
    }

    @Override
    public int findLocalRating(Long customerId, Long localId) {
        customerRepository.findById(customerId).orElseThrow(() -> new CustomerIdNotFoundException(customerId));
        Local local = localService.findLocalById(localId);
        if (!local.getRatings().containsKey(customerId) || local.getRatings().get(customerId) == null) {
            throw new RatingNotFoundException(customerId, localId);
        }
        return local.getRatings().get(customerId);
    }
}
