package mk.ukim.finki.it.reservengo.service.impl;

import mk.ukim.finki.it.reservengo.dto.eventDTO.CreateEventDTO;
import mk.ukim.finki.it.reservengo.dto.eventDTO.DisplayEventDTO;
import mk.ukim.finki.it.reservengo.dto.localDTO.CreateLocalDetailsDTO;
import mk.ukim.finki.it.reservengo.dto.localDTO.DeleteLocalPhotosDTO;
import mk.ukim.finki.it.reservengo.dto.localDTO.DeleteLocalPhotosResultDTO;
import mk.ukim.finki.it.reservengo.model.domain.Event;
import mk.ukim.finki.it.reservengo.model.domain.Local;
import mk.ukim.finki.it.reservengo.model.domain.LocalManager;
import mk.ukim.finki.it.reservengo.model.enumerations.EventStatus;
import mk.ukim.finki.it.reservengo.model.enumerations.EventType;
import mk.ukim.finki.it.reservengo.model.enumerations.Position;
import mk.ukim.finki.it.reservengo.model.exceptions.EventNotInLocalException;
import mk.ukim.finki.it.reservengo.model.exceptions.LocalManagerIdNotFoundException;
import mk.ukim.finki.it.reservengo.model.exceptions.ManagerNotAssignedException;
import mk.ukim.finki.it.reservengo.model.exceptions.UserEmailAlreadyExistsException;
import mk.ukim.finki.it.reservengo.repository.LocalManagerRepository;
import mk.ukim.finki.it.reservengo.service.intf.EventService;
import mk.ukim.finki.it.reservengo.service.intf.LocalManagerService;
import mk.ukim.finki.it.reservengo.service.intf.LocalService;
import mk.ukim.finki.it.reservengo.service.intf.UserService;
import org.springframework.data.domain.Page;
import org.springframework.stereotype.Service;
import org.springframework.web.multipart.MultipartFile;

import java.util.List;
import java.util.stream.Collectors;

@Service
public class LocalManagerServiceImpl implements LocalManagerService {

    private final LocalService localService;
    private final LocalManagerRepository localManagerRepository;
    private final UserService userService;
    private final EventService eventService;

    public LocalManagerServiceImpl(LocalService localService, LocalManagerRepository localManagerRepository, UserService userService, EventService eventService) {
        this.localService = localService;
        this.localManagerRepository = localManagerRepository;
        this.userService = userService;
        this.eventService = eventService;
    }

    @Override
    public void save(LocalManager localManager) {
        if (userService.emailExists(localManager.getEmail())) {
            throw new UserEmailAlreadyExistsException(localManager.getEmail());
        }
        localManagerRepository.save(localManager);
    }

    public List<LocalManager> findAllByLocalId(Long localId) {
        return localManagerRepository.findByLocal_Id(localId);
    }

    @Override
    public List<LocalManager> findAll() {
        return localManagerRepository.findAll();
    }

    @Override
    public LocalManager findManagerById(Long id) {
        return localManagerRepository.findById(id).orElseThrow(() -> new LocalManagerIdNotFoundException(id));
    }

    @Override
    public List<LocalManager> findAllUnassigned() {
        return localManagerRepository.findAll().stream()
                .filter(manager -> manager.getLocal() == null)
                .filter(manager -> manager.getPosition() == null)
                .collect(Collectors.toList());
    }

    @Override
    public void updateLocalAssignment(LocalManager manager, Local local, Position position) {
        manager.setLocal(local);
        manager.setPosition(position);

        localManagerRepository.save(manager);
    }

    @Override
    public Local findLocalByManagerId(Long id) {
        LocalManager localManager = localManagerRepository.findById(id).orElseThrow(() -> new LocalManagerIdNotFoundException(id));
        if (localManager.getLocal() == null) {
            throw new ManagerNotAssignedException(id);
        }
        return localManager.getLocal();
    }

    @Override
    public Local editLocal(Long managerId, CreateLocalDetailsDTO createLocalDetailsDTO) {
        LocalManager manager = localManagerRepository.findById(managerId).orElseThrow(() -> new LocalManagerIdNotFoundException(managerId));
        if (manager.getLocal() == null) {
            throw new ManagerNotAssignedException(manager.getId());
        }
        return localService.edit(manager.getLocal().getId(), createLocalDetailsDTO);
    }

    @Override
    public String uploadLocalLogo(Long managerId, MultipartFile logoFile) {
        Local local = findLocalByManagerId(managerId);

        return localService.addLogo(local.getId(), logoFile);
    }

    @Override
    public void deleteLocalLogo(Long managerId) {
        Local local = findLocalByManagerId(managerId);

        localService.deleteLogo(local.getId());
    }

    @Override
    public String uploadLocalPhoto(Long managerId, MultipartFile photoFile) {
        Local local = findLocalByManagerId(managerId);

        return localService.addPhoto(local.getId(), photoFile);
    }

    @Override
    public DeleteLocalPhotosResultDTO deleteLocalPhoto(Long managerId, DeleteLocalPhotosDTO deleteLocalPhotosDTO) {
        Local local = findLocalByManagerId(managerId);

        return localService.deletePhotos(local.getId(), deleteLocalPhotosDTO.localPhotosUrls());
    }

    @Override
    public void addEvent(Long managerId, CreateEventDTO createEventDTO) {
        Local local = findLocalByManagerId(managerId);
        Event event = createEventDTO.toEvent(local);

        localService.addEvent(local, event);
    }

    @Override
    public void deleteEvent(Long managerId, Long eventId) {
        Local local = findLocalByManagerId(managerId);
        Event event = eventService.findEventById(eventId);

        if (!event.getLocal().getId().equals(local.getId())) {
            throw new EventNotInLocalException(eventId);
        }

        localService.deleteEvent(local, event);
    }

    @Override
    public Page<DisplayEventDTO> findAllEventsByLocal(Long managerId, String name, EventType eventType, EventStatus eventStatus, int page, int size, String sortBy, String direction) {
        Local local = findLocalByManagerId(managerId);
        return eventService.searchEvents(local, name, eventType, eventStatus, page, size, sortBy, direction);
    }

    @Override
    public Event editEvent(Long managerId, Long eventId, CreateEventDTO createEventDTO) {
        Local local = findLocalByManagerId(managerId);
        Event event = eventService.findEventById(eventId);

        if (!event.getLocal().getId().equals(local.getId())) {
            throw new EventNotInLocalException(eventId);
        }

        return localService.editEvent(local, event, createEventDTO);
    }
}
