package mk.ukim.finki.it.reservengo.service.jobs;

import mk.ukim.finki.it.reservengo.model.domain.User;
import mk.ukim.finki.it.reservengo.model.enumerations.Role;
import mk.ukim.finki.it.reservengo.repository.ReservationRepository;
import mk.ukim.finki.it.reservengo.repository.UserRepository;
import org.springframework.scheduling.annotation.Scheduled;
import org.springframework.stereotype.Component;
import org.springframework.transaction.annotation.Transactional;

import java.time.LocalDateTime;
import java.util.List;

@Component
public class ScheduledTasks {

    private final UserRepository userRepository;
    private final ReservationRepository reservationRepository;

    public ScheduledTasks(UserRepository userRepository, ReservationRepository reservationRepository) {
        this.userRepository = userRepository;
        this.reservationRepository = reservationRepository;
    }

    @Transactional
    @Scheduled(cron = "0 0 3 * * ?")
    public void deleteInactiveDisabledUsers() {
        LocalDateTime threshold = LocalDateTime.now().minusDays(30);
        List<User> usersToDelete = userRepository.findAllByEnabledFalseAndLastActivityDateBefore(threshold);
        userRepository.deleteAll(usersToDelete);
    }

    @Transactional
    @Scheduled(cron = "0 0 4 * * ?")
    public void disableInactiveWorkersAndManagers() {
        LocalDateTime threshold = LocalDateTime.now().minusDays(30);
        List<User> usersToDisable = userRepository.findAllByEnabledTrueAndLastActivityDateBefore(threshold)
                .stream()
                .filter(user -> user.getUserRole() == Role.ROLE_LOCAL_MANAGER || user.getUserRole() == Role.ROLE_LOCAL_WORKER)
                .toList();

        usersToDisable.forEach(user -> {
            user.setEnabled(false);
            user.setLastActivityDate(LocalDateTime.now());
        });
        userRepository.saveAll(usersToDisable);
    }

    @Transactional
    @Scheduled(cron = "0 0 0/1 * * ?")
    public void expireOldReservations() {
        reservationRepository.expireOldReservations(LocalDateTime.now());
    }
}
