package mk.ukim.finki.it.reservengo.service.impl;

import mk.ukim.finki.it.reservengo.model.domain.Local;
import mk.ukim.finki.it.reservengo.model.domain.LocalManager;
import mk.ukim.finki.it.reservengo.model.exceptions.LocalIdNotFoundException;
import mk.ukim.finki.it.reservengo.model.exceptions.LogoDeletionException;
import mk.ukim.finki.it.reservengo.model.exceptions.UserEmailAlreadyExistsException;
import mk.ukim.finki.it.reservengo.repository.LocalManagerRepository;
import mk.ukim.finki.it.reservengo.repository.LocalRepository;
import mk.ukim.finki.it.reservengo.service.intf.FileStorageService;
import mk.ukim.finki.it.reservengo.service.intf.LocalManagerService;
import mk.ukim.finki.it.reservengo.service.intf.UserService;
import org.springframework.stereotype.Service;
import org.springframework.web.multipart.MultipartFile;

import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.ArrayList;
import java.util.List;

@Service
public class LocalManagerServiceImpl implements LocalManagerService {

    private final LocalRepository localRepository;
    private final LocalManagerRepository localManagerRepository;
    private final FileStorageService fileStorageService;
    private final UserService userService;

    public LocalManagerServiceImpl(LocalRepository localRepository, LocalManagerRepository localManagerRepository, FileStorageService fileStorageService, UserService userService) {
        this.localRepository = localRepository;
        this.localManagerRepository = localManagerRepository;
        this.fileStorageService = fileStorageService;
        this.userService = userService;
    }

    @Override
    public void uploadLogo(Long localId, MultipartFile logoFile) {
        Local local = localRepository.findById(localId).orElseThrow(() -> new LocalIdNotFoundException(localId));

        String logoPath = fileStorageService.saveLogoFile(logoFile);
        local.setLogoUrl(logoPath);

        localRepository.save(local);
    }

    @Override
    public void save(LocalManager localManager) {
        if (userService.emailExists(localManager.getEmail())) {
            throw new UserEmailAlreadyExistsException(localManager.getEmail());
        }
        localManagerRepository.save(localManager);
    }

    @Override
    public void deleteLocalLogo(Long localId) {
        Local local = localRepository.findById(localId).orElseThrow(() -> new LocalIdNotFoundException(localId));
        String logoUrl = local.getLogoUrl();

        if (logoUrl != null && !logoUrl.isEmpty()) {
            Path filePath = Paths.get("uploads").resolve(Paths.get(logoUrl).getFileName().toString());

            try {
                Files.deleteIfExists(filePath);
            } catch (IOException e) {
                throw new LogoDeletionException("Failed to delete logo from path: " + filePath, e);
            }
        }
        local.setLogoUrl(null);
        localRepository.save(local);
    }

    @Override
    public void uploadPhotos(Long localId, List<MultipartFile> photos) {
        Local local = localRepository.findById(localId).orElseThrow(() -> new LocalIdNotFoundException(localId));

        List<String> photoPaths = local.getLocalPhotos();

        if (photoPaths == null) {
            photoPaths = new ArrayList<>();
        }

        for (MultipartFile photo : photos) {
            if (!photo.isEmpty()) {
                String filePath = fileStorageService.saveLocalPhoto(localId, photo);
                photoPaths.add(filePath);
            }
        }
        local.setLocalPhotos(photoPaths);
        localRepository.save(local);
    }
}
