package mk.ukim.finki.it.reservengo.service.impl;

import mk.ukim.finki.it.reservengo.dto.userDTO.*;
import mk.ukim.finki.it.reservengo.model.domain.User;
import mk.ukim.finki.it.reservengo.model.exceptions.EmailNotFoundException;
import mk.ukim.finki.it.reservengo.model.exceptions.PhotoDeletionException;
import mk.ukim.finki.it.reservengo.model.exceptions.UserEmailAlreadyExistsException;
import mk.ukim.finki.it.reservengo.model.exceptions.UserIdNotFoundException;
import mk.ukim.finki.it.reservengo.repository.UserRepository;
import mk.ukim.finki.it.reservengo.service.intf.FileStorageService;
import mk.ukim.finki.it.reservengo.service.intf.JWTService;
import mk.ukim.finki.it.reservengo.service.intf.UserService;
import org.springframework.security.crypto.password.PasswordEncoder;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.web.multipart.MultipartFile;

@Service
@Transactional
public class UserServiceImpl implements UserService {

    private final UserRepository userRepository;
    private final JWTService jwtService;
    private final PasswordEncoder passwordEncoder;
    private final FileStorageService fileStorageService;

    public UserServiceImpl(UserRepository userRepository, PasswordEncoder passwordEncoder, JWTService jwtService, FileStorageService fileStorageService) {
        this.userRepository = userRepository;
        this.jwtService = jwtService;
        this.passwordEncoder = passwordEncoder;
        this.fileStorageService = fileStorageService;
    }

    @Override
    public User findUserById(Long userId) {
        return userRepository.findById(userId).orElseThrow(() -> new UserIdNotFoundException(userId));
    }

    @Override
    public User findUserByEmail(String email) {
        return userRepository.findByEmail(email).orElseThrow(() -> new EmailNotFoundException(email));
    }

    @Override
    public boolean emailExists(String email) {
        return userRepository.findByEmail(email).isPresent();
    }

    @Override
    public DisplayUserEmailDTO changeEmail(Long userId, EditUserEmailDTO editUserEmailDTO) {
        User user = userRepository.findById(userId).orElseThrow(() -> new UserIdNotFoundException(userId));

        if (editUserEmailDTO.newEmail().equals(user.getEmail())) {
            throw new IllegalArgumentException("New email must be different from the current email.");
        }
        if (emailExists(editUserEmailDTO.newEmail())) {
            throw new UserEmailAlreadyExistsException(editUserEmailDTO.newEmail());
        }
        user.setEmail(editUserEmailDTO.newEmail());

        userRepository.save(user);
        String jwt = jwtService.generateToken(user);

        return DisplayUserEmailDTO.fromUser(user, jwt);
    }

    @Override
    public void changePassword(Long userId, EditUserPasswordDTO editUserPasswordDTO) {
        User user = userRepository.findById(userId).orElseThrow(() -> new UserIdNotFoundException(userId));

        if (!passwordEncoder.matches(editUserPasswordDTO.currentPassword(), user.getPassword())) {
            throw new IllegalArgumentException("Current password is incorrect");
        }

        user.setPassword(passwordEncoder.encode(editUserPasswordDTO.newPassword()));
        userRepository.save(user);
    }

    @Override
    public String uploadProfilePhoto(Long id, MultipartFile photoFile) {
        User user = userRepository.findById(id).orElseThrow(() -> new UserIdNotFoundException(id));

        if (user.getProfilePhotoUrl() != null && !user.getProfilePhotoUrl().isEmpty() ) {
            fileStorageService.deletePhotoFile(user.getProfilePhotoUrl());
        }

        String photoPath = fileStorageService.saveProfilePhotoFile(photoFile);
        user.setProfilePhotoUrl(photoPath);
        userRepository.save(user);

        return photoPath;
    }

    @Override
    public void deleteProfilePhoto(Long id) {
        User user = userRepository.findById(id).orElseThrow(() -> new UserIdNotFoundException(id));
        String photoUrl = user.getProfilePhotoUrl();

        if (photoUrl != null && !photoUrl.isEmpty()) {
            fileStorageService.deletePhotoFile(photoUrl);
            user.setProfilePhotoUrl(null);
            userRepository.save(user);
        } else {
            throw new PhotoDeletionException("No profile photo found for user with id: " + id);
        }
    }

    @Override
    public User editUser(Long userId, EditUserProfileDTO editUserProfileDTO) {
        User user = userRepository.findById(userId).orElseThrow(() -> new UserIdNotFoundException(userId));

        user.setFirstName(editUserProfileDTO.firstName());
        user.setLastName(editUserProfileDTO.lastName());
        user.setPhoneNumber(editUserProfileDTO.phoneNumber());
        userRepository.save(user);

        return user;
    }
}
