package mk.ukim.finki.it.reservengo.service.impl;

import mk.ukim.finki.it.reservengo.dto.jwtDTO.JWTAuthenticationRequestDTO;
import mk.ukim.finki.it.reservengo.dto.jwtDTO.JWTAuthenticationResponseDTO;
import mk.ukim.finki.it.reservengo.dto.jwtDTO.JWTLoginDTO;
import mk.ukim.finki.it.reservengo.model.domain.Customer;
import mk.ukim.finki.it.reservengo.model.domain.LocalManager;
import mk.ukim.finki.it.reservengo.model.domain.LocalWorker;
import mk.ukim.finki.it.reservengo.model.domain.User;
import mk.ukim.finki.it.reservengo.model.enumerations.Role;
import mk.ukim.finki.it.reservengo.model.exceptions.EmailNotFoundException;
import mk.ukim.finki.it.reservengo.repository.CustomerRepository;
import mk.ukim.finki.it.reservengo.repository.LocalManagerRepository;
import mk.ukim.finki.it.reservengo.repository.LocalWorkerRepository;
import mk.ukim.finki.it.reservengo.repository.UserRepository;
import mk.ukim.finki.it.reservengo.service.intf.AuthService;
import mk.ukim.finki.it.reservengo.service.intf.JWTService;
import org.springframework.security.authentication.AuthenticationManager;
import org.springframework.security.authentication.UsernamePasswordAuthenticationToken;
import org.springframework.security.crypto.password.PasswordEncoder;
import org.springframework.stereotype.Service;

@Service
public class AuthServiceImpl implements AuthService {
    private final CustomerRepository customerRepository;
    private final LocalWorkerRepository localWorkerRepository;
    private final LocalManagerRepository localManagerRepository;
    private final JWTService jwtService;
    private final PasswordEncoder passwordEncoder;
    private final AuthenticationManager authenticationManager;
    private final UserRepository userRepository;

    public AuthServiceImpl(CustomerRepository customerRepository, LocalWorkerRepository localWorkerRepository, LocalManagerRepository localManagerRepository, JWTService jwtService, PasswordEncoder passwordEncoder, AuthenticationManager authenticationManager, UserRepository userRepository) {
        this.customerRepository = customerRepository;
        this.localWorkerRepository = localWorkerRepository;
        this.localManagerRepository = localManagerRepository;
        this.jwtService = jwtService;
        this.passwordEncoder = passwordEncoder;
        this.authenticationManager = authenticationManager;
        this.userRepository = userRepository;
    }

    @Override
    public JWTAuthenticationResponseDTO registerCustomer(JWTAuthenticationRequestDTO jwtAuthenticationRequestDTO) {

        Customer customer = new Customer(
                jwtAuthenticationRequestDTO.firstName(),
                jwtAuthenticationRequestDTO.lastName(),
                jwtAuthenticationRequestDTO.email(),
                passwordEncoder.encode(jwtAuthenticationRequestDTO.password()),
                jwtAuthenticationRequestDTO.phoneNumber(),
                Role.ROLE_CUSTOMER
        );

        customerRepository.save(customer);
        String jwt = jwtService.generateToken(customer);

        return new JWTAuthenticationResponseDTO(
                customer.getId(),
                customer.getFirstName(),
                customer.getLastName(),
                customer.getEmail(),
                customer.getPhoneNumber(),
                Role.ROLE_CUSTOMER,
                jwt);
    }

    @Override
    public JWTAuthenticationResponseDTO registerLocalWorker(JWTAuthenticationRequestDTO jwtAuthenticationRequestDTO) {
        LocalWorker localWorker = new LocalWorker(
                jwtAuthenticationRequestDTO.firstName(),
                jwtAuthenticationRequestDTO.lastName(),
                jwtAuthenticationRequestDTO.email(),
                passwordEncoder.encode(jwtAuthenticationRequestDTO.password()),
                jwtAuthenticationRequestDTO.phoneNumber(),
                Role.ROLE_LOCAL_WORKER
        );
        localWorkerRepository.save(localWorker);
        String jwt = jwtService.generateToken(localWorker);

        return new JWTAuthenticationResponseDTO(
                localWorker.getId(),
                localWorker.getFirstName(),
                localWorker.getLastName(),
                localWorker.getEmail(),
                localWorker.getPhoneNumber(),
                Role.ROLE_LOCAL_WORKER,
                jwt);
    }

    @Override
    public JWTAuthenticationResponseDTO registerLocalManager(JWTAuthenticationRequestDTO jwtAuthenticationRequestDTO) {
        LocalManager localManager = new LocalManager(
                jwtAuthenticationRequestDTO.firstName(),
                jwtAuthenticationRequestDTO.lastName(),
                jwtAuthenticationRequestDTO.email(),
                passwordEncoder.encode(jwtAuthenticationRequestDTO.password()),
                jwtAuthenticationRequestDTO.phoneNumber(),
                Role.ROLE_LOCAL_MANAGER
        );
        localManagerRepository.save(localManager);
        String jwt = jwtService.generateToken(localManager);

        return new JWTAuthenticationResponseDTO(
                localManager.getId(),
                localManager.getFirstName(),
                localManager.getLastName(),
                localManager.getEmail(),
                localManager.getPhoneNumber(),
                Role.ROLE_LOCAL_MANAGER,
                jwt);
    }

    @Override
    public JWTAuthenticationResponseDTO login(JWTLoginDTO jwtLoginDTO) {
        authenticationManager.authenticate(new UsernamePasswordAuthenticationToken(jwtLoginDTO.email(), jwtLoginDTO.password()));
        User user = userRepository.findByEmail(jwtLoginDTO.email()).orElseThrow(() -> new EmailNotFoundException(jwtLoginDTO.email()));
        String jwt = jwtService.generateToken(user);

        return new JWTAuthenticationResponseDTO(
                user.getId(),
                user.getFirstName(),
                user.getLastName(),
                user.getEmail(),
                user.getPhoneNumber(),
                user.getUserRole(),
                jwt);
    }
}
