package mk.ukim.finki.it.reservengo.service.impl;

import mk.ukim.finki.it.reservengo.dto.eventDTO.DisplayEventDTO;
import mk.ukim.finki.it.reservengo.model.domain.Event;
import mk.ukim.finki.it.reservengo.model.domain.Local;
import mk.ukim.finki.it.reservengo.model.enumerations.EventStatus;
import mk.ukim.finki.it.reservengo.model.enumerations.EventType;
import mk.ukim.finki.it.reservengo.model.exceptions.EventIdNotFoundException;
import mk.ukim.finki.it.reservengo.repository.EventRepository;
import mk.ukim.finki.it.reservengo.service.intf.EventService;
import org.springframework.data.domain.*;
import org.springframework.data.jpa.domain.Specification;
import org.springframework.stereotype.Service;

import java.util.List;

@Service
public class EventServiceImpl implements EventService {
    private final EventRepository eventRepository;

    public EventServiceImpl(EventRepository eventRepository) {
        this.eventRepository = eventRepository;
    }

    @Override
    public Page<DisplayEventDTO> searchEvents(Local local, String name, EventType eventType, EventStatus eventStatus, int page, int size, String sortBy, String direction) {
        boolean needsManualProcessing;
        needsManualProcessing = eventStatus != null;

        Specification<Event> spec = Specification.where(null);

        if (local != null) {
            spec = spec.and((root, query, cb) ->
                    cb.equal(root.get("local"), local));
        }

        if (name != null && !name.trim().isEmpty()) {
            spec = spec.and((root, query, cb) ->
                    cb.like(cb.lower(root.get("name")), "%" + name.toLowerCase() + "%"));
        }

        if (eventType != null) {
            spec = spec.and((root, query, cb) ->
                    cb.equal(root.get("eventType"), eventType));
        }

        Pageable pageable = needsManualProcessing
                ? Pageable.unpaged()
                : PageRequest.of(page, size, getSort(sortBy, direction));

        Page<Event> eventsPage = eventRepository.findAll(spec, pageable);

        // Manual status filtering
        if (needsManualProcessing) {
            List<DisplayEventDTO> filtered = eventsPage.getContent().stream()
                    .filter(event -> event.getStatus() == eventStatus)
                    .map(DisplayEventDTO::fromEvent)
                    .toList();

            int total = filtered.size();
            int start = Math.min(page * size, total);
            int end = Math.min(start + size, total);
            List<DisplayEventDTO> paged = filtered.subList(start, end);

            return new PageImpl<>(paged, PageRequest.of(page, size), total);
        }

        return eventsPage.map(DisplayEventDTO::fromEvent);
    }


    @Override
    public Event findEventById(Long id) {
        return eventRepository.findById(id).orElseThrow(() -> new EventIdNotFoundException(id));
    }

    @Override
    public Event save(Event event) {
        return eventRepository.save(event);
    }

    private Sort getSort(String sortBy, String direction) {
        Sort.Direction sortDirection = "desc".equalsIgnoreCase(direction) ? Sort.Direction.DESC : Sort.Direction.ASC;

        return switch (sortBy) {
            case "createdAt" -> Sort.by(sortDirection, "createdAt");
            default -> Sort.by(sortDirection, "name");
        };
    }
}
