package mk.ukim.finki.it.reservengo.web.controller;

import mk.ukim.finki.it.reservengo.dto.customerDTO.RatingDTO;
import mk.ukim.finki.it.reservengo.dto.eventDTO.DisplayEventDTO;
import mk.ukim.finki.it.reservengo.dto.localDTO.DisplayLocalDTO;
import mk.ukim.finki.it.reservengo.dto.reservationDTO.*;
import mk.ukim.finki.it.reservengo.model.domain.User;
import mk.ukim.finki.it.reservengo.service.intf.CustomerService;
import org.springframework.http.HttpStatus;
import org.springframework.http.ResponseEntity;
import org.springframework.security.access.prepost.PreAuthorize;
import org.springframework.security.core.annotation.AuthenticationPrincipal;
import org.springframework.web.bind.annotation.*;

import java.util.List;

@RestController
@RequestMapping("/api/customer")
@PreAuthorize("hasRole('CUSTOMER')")
public class CustomerController {

    private final CustomerService customerService;

    public CustomerController(CustomerService customerService) {
        this.customerService = customerService;
    }

    @GetMapping("/favourite-locals")
    public ResponseEntity<List<DisplayLocalDTO>> listFavouriteLocals(@AuthenticationPrincipal User user) {
        List<DisplayLocalDTO> list = customerService.listFavouriteLocals(user.getId());
        return new ResponseEntity<>(list, HttpStatus.OK);
    }

    @PostMapping("/favourite-locals/add/{id}")
    public ResponseEntity<Void> addFavouriteLocal(@AuthenticationPrincipal User user, @PathVariable Long id) {
        customerService.addFavouriteLocal(user.getId(), id);
        return new ResponseEntity<>(HttpStatus.OK);
    }

    @PostMapping("/favourite-locals/remove/{id}")
    public ResponseEntity<Void> removeFavouriteLocal(@AuthenticationPrincipal User user, @PathVariable Long id) {
        customerService.removeFavouriteLocal(user.getId(), id);
        return new ResponseEntity<>(HttpStatus.OK);
    }

    @GetMapping("/favourite-events")
    public ResponseEntity<List<DisplayEventDTO>> listFavouriteEvents(@AuthenticationPrincipal User user) {
        List<DisplayEventDTO> list = customerService.listFavouriteEvents(user.getId());
        return new ResponseEntity<>(list, HttpStatus.OK);
    }

    @PostMapping("/favourite-events/add/{id}")
    public ResponseEntity<Void> addFavouriteEvent(@AuthenticationPrincipal User user, @PathVariable Long id) {
        customerService.addFavouriteEvent(user.getId(), id);
        return new ResponseEntity<>(HttpStatus.OK);
    }

    @PostMapping("/favourite-events/remove/{id}")
    public ResponseEntity<Void> removeFavouriteEvent(@AuthenticationPrincipal User user, @PathVariable Long id) {
        customerService.removeFavouriteEvent(user.getId(), id);
        return new ResponseEntity<>(HttpStatus.OK);
    }

    @PutMapping("/local/{id}/rate")
    public ResponseEntity<Void> rateLocal(@AuthenticationPrincipal User user, @PathVariable Long id, @RequestBody RatingDTO ratingDTO) {
        customerService.rateLocal(user.getId(), id, ratingDTO);
        return new ResponseEntity<>(HttpStatus.OK);
    }

    @DeleteMapping("/local/{id}/remove-rating")
    public ResponseEntity<Void> deleteRating(@AuthenticationPrincipal User user, @PathVariable Long id) {
        customerService.removeRating(user.getId(), id);
        return new ResponseEntity<>(HttpStatus.OK);
    }

    @GetMapping("/local/{id}/rating")
    public ResponseEntity<RatingDTO> getRatingForLocal(@AuthenticationPrincipal User user, @PathVariable Long id) {
        RatingDTO ratingDTO = RatingDTO.from(customerService.findLocalRating(user.getId(), id));
        return new ResponseEntity<>(ratingDTO, HttpStatus.OK);
    }

    @GetMapping("/reservations")
    public ResponseEntity<List<DisplayReservationDTO>> listCustomerReservations(@AuthenticationPrincipal User user) {
        List<DisplayReservationDTO> allReservation = customerService.listAllReservations(user.getId());
        return new ResponseEntity<>(allReservation, HttpStatus.OK);
    }

    @GetMapping("/reservations/{reservationId}")
    public ResponseEntity<DisplayReservationDTO> getReservationDetails(@AuthenticationPrincipal User user, @PathVariable Long reservationId) {
        DisplayReservationDTO reservationDTO = customerService.getReservationPreview(user.getId(), reservationId);
        return new ResponseEntity<>(reservationDTO, HttpStatus.OK);
    }

    @PutMapping("/reservations/{reservationId}/cancel")
    public ResponseEntity<Void> cancelReservation(@AuthenticationPrincipal User user, @PathVariable Long reservationId) {
        customerService.cancelReservation(user.getId(), reservationId);
        return new ResponseEntity<>(HttpStatus.OK);
    }

    @GetMapping("/reservations/{reservationId}/edit-info")
    public ResponseEntity<EditReservationInfoDTO> editReservationInfo(@AuthenticationPrincipal User user, @PathVariable Long reservationId) {
        EditReservationInfoDTO editReservationInfoDTO = customerService.getEditReservationInfo(user.getId(), reservationId);
        return new ResponseEntity<>(editReservationInfoDTO, HttpStatus.OK);
    }

    @PutMapping("/reservations/{reservationId}/edit")
    public ResponseEntity<DisplayReservationDTO> editReservation(@AuthenticationPrincipal User user, @PathVariable Long reservationId, @RequestBody EditReservationDTO editReservationDTO) {
        DisplayReservationDTO editedReservationDTO = customerService.editReservation(user.getId(), reservationId, editReservationDTO);
        return new ResponseEntity<>(editedReservationDTO, HttpStatus.OK);
    }

    @PostMapping("/reservations/create")
    public ResponseEntity<DisplayReservationDTO> makeReservation(@AuthenticationPrincipal User user, @RequestBody CreateReservationDTO createReservationDTO) {
        DisplayReservationDTO newReservation = customerService.makeReservation(user.getId(), createReservationDTO);
        return new ResponseEntity<>(newReservation, HttpStatus.OK);
    }

    @DeleteMapping("/reservations/delete")
    public ResponseEntity<Void> deleteReservation(@AuthenticationPrincipal User user, @RequestBody DeleteReservationDTO deleteReservationDTO) {
        customerService.deleteReservation(user.getId(), deleteReservationDTO);
        return new ResponseEntity<>(HttpStatus.OK);
    }
}