package mk.ukim.finki.it.reservengo.service.impl;

import mk.ukim.finki.it.reservengo.dto.localDTO.CreateLocalDetailsDTO;
import mk.ukim.finki.it.reservengo.dto.localDTO.DeleteLocalPhotosResultDTO;
import mk.ukim.finki.it.reservengo.dto.localDTO.DisplayLocalDTO;
import mk.ukim.finki.it.reservengo.model.domain.Local;
import mk.ukim.finki.it.reservengo.model.enumerations.LocalType;
import mk.ukim.finki.it.reservengo.model.enumerations.ProvidedService;
import mk.ukim.finki.it.reservengo.model.exceptions.*;
import mk.ukim.finki.it.reservengo.repository.LocalRepository;
import mk.ukim.finki.it.reservengo.service.intf.FileStorageService;
import mk.ukim.finki.it.reservengo.service.intf.LocalService;
import org.springframework.data.domain.*;
import org.springframework.data.jpa.domain.Specification;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.web.multipart.MultipartFile;

import java.util.ArrayList;
import java.util.Comparator;
import java.util.List;
import java.util.stream.Collectors;

@Service
public class LocalServiceImpl implements LocalService {

    private final LocalRepository localRepository;
    private final FileStorageService fileStorageService;

    public LocalServiceImpl(LocalRepository localRepository, FileStorageService fileStorageService) {
        this.localRepository = localRepository;
        this.fileStorageService = fileStorageService;
    }

    @Override
    public List<Local> listAll() {
        return localRepository.findAll();
    }

    @Override
    public Local findLocalById(Long id) {
        return localRepository.findById(id).orElseThrow(() -> new LocalIdNotFoundException(id));
    }

    @Override
    public void save(String name) {
        Local local = new Local(name);
        localRepository.save(local);
    }

    @Override
    @Transactional
    public Local edit(Long localId, CreateLocalDetailsDTO createLocalDetailsDTO) {
        Local local = localRepository.findById(localId).orElseThrow(() -> new LocalIdNotFoundException(localId));

        local.setName(createLocalDetailsDTO.name());
        local.setDescription(createLocalDetailsDTO.description());
        local.setAddress(createLocalDetailsDTO.address());
        local.setWorkingHours(createLocalDetailsDTO.workingHours());
        local.setAvailableServices(createLocalDetailsDTO.services());
        local.setLocalType(createLocalDetailsDTO.localType());
        local.setMenuLink(createLocalDetailsDTO.menuLink());
        local.setContact(createLocalDetailsDTO.contact().toContact());

        localRepository.save(local);

        return local;
    }

    @Override
    public void delete(Long id) {
        Local local = localRepository.findById(id).orElseThrow(() -> new LocalIdNotFoundException(id));
        localRepository.delete(local);
    }

    @Override
    public String addLogo(Long localId, MultipartFile multipartFile) {
        Local local = localRepository.findById(localId).orElseThrow(() -> new LocalIdNotFoundException(localId));

        if (local.getLogoUrl() != null && !local.getLogoUrl().isEmpty()) {
            fileStorageService.deletePhotoFile(local.getLogoUrl());
        }

        String logoPath = fileStorageService.saveLogoFile(multipartFile);
        local.setLogoUrl(logoPath);
        localRepository.save(local);

        return logoPath;
    }

    @Override
    @Transactional
    public void deleteLogo(Long localId) {
        Local local = localRepository.findById(localId).orElseThrow(() -> new LocalIdNotFoundException(localId));
        String logoUrl = local.getLogoUrl();

        if (logoUrl != null && !logoUrl.isEmpty()) {
            fileStorageService.deletePhotoFile(logoUrl);
            local.setLogoUrl(null);
            localRepository.save(local);
        } else {
            throw new PhotoDeletionException("No logo found for local with id: " + localId);
        }
    }

    @Override
    public String addPhoto(Long localId, MultipartFile photoFile) {
        Local local = localRepository.findById(localId).orElseThrow(() -> new LocalIdNotFoundException(localId));

        List<String> photoPaths = local.getLocalPhotos();

        if (photoPaths == null) {
            photoPaths = new ArrayList<>();
        }

        String photoPath = fileStorageService.savePhotoFile(photoFile);
        photoPaths.add(photoPath);
        localRepository.save(local);

        return photoPath;
    }

    @Override
    @Transactional
    public DeleteLocalPhotosResultDTO deletePhotos(Long localId, List<String> photoPaths) {
        Local local = localRepository.findById(localId).orElseThrow(() -> new LocalIdNotFoundException(localId));
        List<String> currentPhotoUrls = local.getLocalPhotos();

        if (currentPhotoUrls == null || currentPhotoUrls.isEmpty()) {
            throw new PhotoDeletionException("No photos found for local with id: " + localId);
        }

        List<String> deletedPhotos = new ArrayList<>();
        List<String> notFoundPhotos = new ArrayList<>();

        for (String photoUrl : photoPaths) {
            if (currentPhotoUrls.contains(photoUrl)) {
                fileStorageService.deletePhotoFile(photoUrl);
                currentPhotoUrls.remove(photoUrl);
                deletedPhotos.add(photoUrl);
            } else {
                notFoundPhotos.add(photoUrl);
            }
        }
        local.setLocalPhotos(currentPhotoUrls);
        localRepository.save(local);

        return new DeleteLocalPhotosResultDTO(deletedPhotos, notFoundPhotos);
    }

    @Override
    public Page<DisplayLocalDTO> searchLocals(String name, List<ProvidedService> services, LocalType localType, int page, int size, String sortBy, String direction) {
        boolean sortByRating = "rating".equalsIgnoreCase(sortBy);

        Pageable pageable = PageRequest.of(page, size, getSort(sortBy, direction));

        Specification<Local> spec = Specification.where(null);

        if (name != null && !name.trim().isEmpty()) {
            spec = spec.and((root, query, cb) ->
                    cb.like(cb.lower(root.get("name")), "%" + name.toLowerCase() + "%"));
        }

        if (services != null && !services.isEmpty()) {
            spec = spec.and((root, query, cb) -> root.join("availableServices").in(services));
        }

        if (localType != null) {
            spec = spec.and((root, query, cb) -> cb.equal(root.get("localType"), localType));
        }

        Page<Local> localsPage = localRepository.findAll(spec, pageable);

        // If sorting by rating, do it in-memory after mapping to DTOs
        if (sortByRating) {
            List<DisplayLocalDTO> sorted = localsPage.getContent().stream()
                    .map(DisplayLocalDTO::fromLocal)
                    .sorted(Comparator.comparingDouble(DisplayLocalDTO::averageRating).reversed())
                    .collect(Collectors.toList());

            return new PageImpl<>(sorted, pageable, localsPage.getTotalElements());
        }

        return localsPage.map(DisplayLocalDTO::fromLocal);
    }

    private Sort getSort(String sortBy, String direction) {
        Sort.Direction sortDirection = "desc".equalsIgnoreCase(direction) ? Sort.Direction.DESC : Sort.Direction.ASC;

        return switch (sortBy) {
            case "createdAt" -> Sort.by(sortDirection, "createdAt");
            case "rating" -> Sort.unsorted(); // handled manually
            default -> Sort.by(sortDirection, "name");
        };
    }
}
