package mk.ukim.finki.it.reservengo.service.impl;

import mk.ukim.finki.it.reservengo.dto.customerDTO.RatingDTO;
import mk.ukim.finki.it.reservengo.dto.eventDTO.DisplayEventDTO;
import mk.ukim.finki.it.reservengo.dto.localDTO.DisplayLocalDTO;
import mk.ukim.finki.it.reservengo.dto.reservationDTO.CreateReservationDTO;
import mk.ukim.finki.it.reservengo.dto.reservationDTO.DisplayReservationDTO;
import mk.ukim.finki.it.reservengo.dto.reservationDTO.EditReservationDTO;
import mk.ukim.finki.it.reservengo.dto.reservationDTO.EditReservationInfoDTO;
import mk.ukim.finki.it.reservengo.dto.tableDTO.TableInfoDTO;
import mk.ukim.finki.it.reservengo.model.domain.*;
import mk.ukim.finki.it.reservengo.model.enumerations.ReservationStatus;
import mk.ukim.finki.it.reservengo.model.exceptions.*;
import mk.ukim.finki.it.reservengo.repository.CustomerRepository;
import mk.ukim.finki.it.reservengo.repository.ReservationRepository;
import mk.ukim.finki.it.reservengo.repository.TableRepository;
import mk.ukim.finki.it.reservengo.service.intf.*;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.time.LocalDateTime;
import java.util.List;

@Service
public class CustomerServiceImpl implements CustomerService {
    private final CustomerRepository customerRepository;
    private final LocalService localService;
    private final UserService userService;
    private final EventService eventService;
    private final ReservationRepository reservationRepository;
    private final TableRepository tableRepository;
    private final TableService tableService;

    public CustomerServiceImpl(CustomerRepository customerRepository, LocalService localService, UserService userService, EventService eventService, ReservationRepository reservationRepository, TableRepository tableRepository, TableService tableService) {
        this.customerRepository = customerRepository;
        this.localService = localService;
        this.userService = userService;
        this.eventService = eventService;
        this.reservationRepository = reservationRepository;
        this.tableRepository = tableRepository;
        this.tableService = tableService;
    }

    @Override
    public List<DisplayLocalDTO> listFavouriteLocals(Long id) {
        Customer customer = customerRepository.findById(id).orElseThrow(() -> new CustomerIdNotFoundException(id));
        return DisplayLocalDTO.fromLocals(customer.getFavouriteLocals());
    }

    @Override
    public void addFavouriteLocal(Long userId, Long localId) {
        Customer customer = customerRepository.findById(userId).orElseThrow(() -> new CustomerIdNotFoundException(userId));
        Local local = localService.findLocalById(localId);

        if (customer.getFavouriteLocals().contains(local)) {
            throw new LocalAlreadyFavouredException(localId);
        }

        customer.getFavouriteLocals().add(local);
        customerRepository.save(customer);
    }

    @Override
    public void removeFavouriteLocal(Long userId, Long localId) {
        Customer customer = customerRepository.findById(userId).orElseThrow(() -> new CustomerIdNotFoundException(userId));
        Local local = localService.findLocalById(localId);

        if (!customer.getFavouriteLocals().contains(local)) {
            throw new LocalAlreadyUnfavouredException(localId);
        }

        customer.getFavouriteLocals().remove(local);
        customerRepository.save(customer);
    }

    @Override
    public void save(Customer customer) {
        if (userService.emailExists(customer.getEmail())) {
            throw new UserEmailAlreadyExistsException(customer.getEmail());
        }
        customerRepository.save(customer);
    }

    @Override
    public List<DisplayEventDTO> listFavouriteEvents(Long id) {
        Customer customer = customerRepository.findById(id).orElseThrow(() -> new CustomerIdNotFoundException(id));
        return DisplayEventDTO.fromEvents(customer.getFavouriteEvents());
    }

    @Override
    public void addFavouriteEvent(Long userId, Long eventId) {
        Customer customer = customerRepository.findById(userId).orElseThrow(() -> new CustomerIdNotFoundException(userId));
        Event event = eventService.findEventById(eventId);

        if (customer.getFavouriteEvents().contains(event)) {
            throw new EventAlreadyFavouredException(eventId);
        }

        customer.getFavouriteEvents().add(event);
        customerRepository.save(customer);
    }

    @Override
    public void removeFavouriteEvent(Long userId, Long eventId) {
        Customer customer = customerRepository.findById(userId).orElseThrow(() -> new CustomerIdNotFoundException(userId));
        Event event = eventService.findEventById(eventId);

        if (!customer.getFavouriteEvents().contains(event)) {
            throw new EventAlreadyUnfavouredException(eventId);
        }

        customer.getFavouriteEvents().remove(event);
        customerRepository.save(customer);
    }

    @Override
    public void rateLocal(Long customerId, Long localId, RatingDTO ratingDTO) {
        customerRepository.findById(customerId).orElseThrow(() -> new CustomerIdNotFoundException(customerId));
        localService.addRating(customerId, localId, ratingDTO.rating());
    }

    @Override
    public void removeRating(Long customerId, Long localId) {
        customerRepository.findById(customerId).orElseThrow(() -> new CustomerIdNotFoundException(customerId));
        localService.removeRating(customerId, localId);
    }

    @Override
    public int findLocalRating(Long customerId, Long localId) {
        customerRepository.findById(customerId).orElseThrow(() -> new CustomerIdNotFoundException(customerId));
        Local local = localService.findLocalById(localId);
        if (!local.getRatings().containsKey(customerId) || local.getRatings().get(customerId) == null) {
            throw new RatingNotFoundException(customerId, localId);
        }
        return local.getRatings().get(customerId);
    }

    @Override
    public List<DisplayReservationDTO> listAllReservations(Long id) {
        Customer customer = customerRepository.findById(id).orElseThrow(() -> new CustomerIdNotFoundException(id));
        return DisplayReservationDTO.from(customer.getReservations());
    }

    @Override
    public DisplayReservationDTO getReservationPreview(Long customerId, Long reservationId) {
        Reservation reservation = reservationRepository.findById(reservationId).orElseThrow(() -> new ReservationIdNotFoundException(reservationId));

        if (!reservation.getCustomer().getId().equals(customerId)) {
            throw new UnauthorizedReservationAccessException(customerId, reservationId);
        }

        return DisplayReservationDTO.from(reservation);
    }

    @Override
    @Transactional
    public DisplayReservationDTO editReservation(Long id, Long reservationId, EditReservationDTO editReservationDTO) {
        Reservation reservation = reservationRepository.findById(reservationId).orElseThrow(() -> new ReservationIdNotFoundException(reservationId));

        if (!reservation.getCustomer().getId().equals(id)) {
            throw new UnauthorizedReservationAccessException(id, reservationId);
        }

        LocalDateTime newTime = editReservationDTO.timeOfReservation();
        Long newTableId = editReservationDTO.tableId();

        if (!tableService.isTableAvailable(newTableId, newTime)) {
            throw new TableNotAvailableException(newTableId, newTime);
        }

        reservation.setTimeOfReservation(newTime);

        DiningTable table = tableRepository.findById(newTableId).orElseThrow(() -> new TableIdNotFoundException(editReservationDTO.tableId()));

        reservation.setTable(table);
        reservation.setStatus(editReservationDTO.status());
        reservationRepository.save(reservation);

        return DisplayReservationDTO.from(reservation);
    }

    @Override
    public void cancelReservation(Long id, Long reservationId) {
        Reservation reservation = reservationRepository.findById(reservationId).orElseThrow(() -> new ReservationIdNotFoundException(reservationId));

        if (!reservation.getCustomer().getId().equals(id)) {
            throw new UnauthorizedReservationAccessException(id, reservationId);
        }

        reservation.setStatus(ReservationStatus.CANCELED);
        reservationRepository.save(reservation);
    }

    @Override
    public EditReservationInfoDTO getEditReservationInfo(Long id, Long reservationId) {
        Reservation reservation = reservationRepository.findById(reservationId).orElseThrow(() -> new ReservationIdNotFoundException(reservationId));

        if (!reservation.getCustomer().getId().equals(id)) {
            throw new UnauthorizedReservationAccessException(id, reservationId);
        }

        List<TableInfoDTO> availableTablesAtThatTime = tableService.getAvailableTables(reservation.getLocal().getId(), reservation.getTimeOfReservation());

        return EditReservationInfoDTO.from(reservation, availableTablesAtThatTime);
    }

    @Override
    @Transactional
    public DisplayReservationDTO makeReservation(Long id, CreateReservationDTO createReservationDTO) {
        Customer customer = customerRepository.findById(id).orElseThrow(() -> new CustomerIdNotFoundException(id));
        Local local = localService.findLocalById(createReservationDTO.localId());
        Long tableId = createReservationDTO.tableId();

        if (!tableService.isTableAvailable(tableId, createReservationDTO.timeOfReservation())) {
            throw new TableNotAvailableException(tableId, createReservationDTO.timeOfReservation());
        }

        DiningTable table = tableRepository.findById(tableId).orElseThrow(() -> new TableIdNotFoundException(tableId));

        Reservation reservation = new Reservation();
        reservation.setCustomer(customer);
        reservation.setLocal(local);
        reservation.setTimeOfReservation(createReservationDTO.timeOfReservation());
        reservation.setTable(table);
        reservation.setStatus(ReservationStatus.PENDING);

        reservationRepository.save(reservation);

        return DisplayReservationDTO.from(reservation);
    }
}
