import React, {useState, useEffect, useContext} from 'react';
import 'bootstrap/dist/css/bootstrap.min.css';
import StarRating from './StarRating';
import { useNavigate } from 'react-router-dom';
import {RestaurantContext} from "./RestaurantContext";
import {Alert} from "bootstrap";

const parseTime = (timeString) => {
    const [hours, minutes] = timeString.trim().split(':').map(Number);
    return new Date().setHours(hours, minutes, 0, 0);
};

const roundToNextQuarter = (date) => {
    const minutes = date.getMinutes();
    const roundedMinutes = Math.floor(minutes / 15) * 15;
    date.setMinutes(roundedMinutes, 0, 0);
    return date;
};

const shouldMoveToNextDay = (currentTime, endTime) => {
    return (endTime - currentTime) <= 2 * 60 * 60 * 1000;
};

const Restaurants = () => {
    const [restaurants, setRestaurants] = useState([]);
    const navigate = useNavigate();
    const restaurantContext = useContext(RestaurantContext);
    console.log(restaurantContext)

    useEffect(() => {
        setRestaurants(restaurantContext.restaurants);
    }, [restaurantContext]);


    const generateTimeSlots = (operatingHours) => {
        const timeSlots = [];
        const [startTimeStr, endTimeStr] = operatingHours.split('-').map((time) => time.trim());

        const startTime = parseTime(startTimeStr);
        let endTime = parseTime(endTimeStr);

        const currentTime = new Date().getTime();
        if (shouldMoveToNextDay(currentTime, endTime)) {
            endTime += 24 * 60 * 60 * 1000;
        }

        let currentTimeSlot = new Date(startTime);
        currentTimeSlot = roundToNextQuarter(currentTimeSlot);

        while (currentTimeSlot.getTime() < endTime) {
            timeSlots.push(currentTimeSlot.toISOString());
            currentTimeSlot.setMinutes(currentTimeSlot.getMinutes() + 15);
        }

        return timeSlots;
    };

    const handleDetailClick = (restaurantId) => {
        navigate(`/restaurants/${restaurantId}`);
    };

    const handleTimeSlotClick = (table, timeSlot, restaurant) => {
        const tableNumber = table.id;
        const formattedTimeSlot = timeSlot;
        const restaurantId = restaurant.restaurantId;

        const encodedTableNumber = encodeURIComponent(tableNumber);
        const encodedTimeSlot = encodeURIComponent(formattedTimeSlot);
        const encodedRestaurantId = encodeURIComponent(restaurantId);

        navigate(`/reservationConfirmation/${encodedTableNumber}/${encodedTimeSlot}/${encodedRestaurantId}`);
    };

    const renderTimeSlots = (tablesList, restaurant) => {
        const currentTime = new Date().getTime();
        let renderedTimeSlots = {};

        if (tablesList.length === 0) {
            return <p>No tables available for reservations at this restaurant.</p>;
        }

        return tablesList.flatMap((table) => {
            const tableTimeSlots = generateTimeSlots(restaurant.operatingHours);

            if (!renderedTimeSlots[table.capacity]) {
                renderedTimeSlots[table.capacity] = 0;
                return (
                    <div key={table.capacity}>
                        <h3>Table for {table.capacity} guests</h3>
                        {tableTimeSlots.map((timeSlot, index) => {
                            const timeSlotTime = new Date(timeSlot).getTime();

                            if (timeSlotTime > currentTime && renderedTimeSlots[table.capacity] < 3) {
                                renderedTimeSlots[table.capacity]++;
                                const timeSlotDateTime = new Date(timeSlot);
                                const formattedTime = timeSlotDateTime.toLocaleTimeString([], {
                                    hour: '2-digit',
                                    minute: '2-digit'
                                });

                                return (
                                    <button
                                        key={index}
                                        className="btn btn-primary me-2 mb-2"
                                        onClick={() => handleTimeSlotClick(table, timeSlot, restaurant)}
                                    >
                                        {formattedTime} {}
                                    </button>
                                );
                            } else {
                                return null;
                            }
                        })}
                    </div>
                );
            } else {
                return null;
            }
        });
    };

    return (
        <div className="container">
            <h2>Restaurants</h2>
            <div className="row">
                {restaurants.map((restaurant) => (
                    <div key={restaurant.restaurantId} className="col-md-4 mb-4">
                        <div className="card">
                            <div className="card-body">
                                <h5 className="card-title">
                                    {restaurant.name}
                                </h5>
                                <p className="card-text">{restaurant.cuisineType}</p>
                                <p className="card-text">{restaurant.operatingHours}</p>
                                <p className="card-text">Ul. {restaurant.address}</p>

                                {restaurant.tablesList && restaurant.tablesList.length > 0 ? (
                                    <div className="d-flex flex-wrap">
                                        {renderTimeSlots(restaurant.tablesList, restaurant)}
                                    </div>
                                ) : (
                                    <p>No tables available for reservations at this restaurant.</p>
                                )}
                            </div>
                            <button onClick={() => handleDetailClick(restaurant.restaurantId)} className="btn btn-primary">
                                View Details
                            </button>
                        </div>
                    </div>
                ))}
            </div>
        </div>
    );
};

export default Restaurants;
