import React, { useState, useEffect } from 'react';
import { useNavigate } from 'react-router-dom';
import axios from 'axios';
import 'bootstrap/dist/css/bootstrap.min.css';
import { useParams } from 'react-router-dom';
import StarRating from "./StarRating";

const RestaurantDetails = () => {
    const navigate = useNavigate();

    const { id } = useParams();
    const [restaurant, setRestaurant] = useState(null);
    const [selectedTableId, setSelectedTableId] = useState('');
    const [selectedTimeSlot, setSelectedTimeSlot] = useState('');
    const [selectedTable, setSelectedTable] = useState(null);
    const [filteredTimeSlots, setFilteredTimeSlots] = useState([]);
    const [selectedCapacity, setSelectedCapacity] = useState(''); // Define selectedCapacity state

    useEffect(() => {
        const fetchRestaurantDetails = async () => {
            try {
                if (!id) return;
                const response = await axios.get(`http://localhost:8080/api/restaurants/${id}`);
                setRestaurant(response.data);
            } catch (error) {
                console.error('Error fetching restaurant details:', error);
            }
        };

        fetchRestaurantDetails();
    }, [id]);

    useEffect(() => {
        if (!selectedTableId) return; // If no table is selected, return early

        const fetchTableDetails = async () => {
            try {
                const response = await axios.get(`http://localhost:8080/api/tables/${selectedTableId}`);
                setSelectedTable(response.data);
            } catch (error) {
                console.error('Error fetching table details:', error);
            }
        };

        fetchTableDetails();
    }, [selectedTableId]);

    useEffect(() => {
        if (!selectedTable || !restaurant) return; // If table or restaurant is not loaded, return early

        // Filter time slots based on the selected table
        const currentTime = new Date();
        const filteredSlots = selectedTable.timeSlots.filter(timeSlot => new Date(timeSlot) >= currentTime);
        setFilteredTimeSlots(filteredSlots);
    }, [selectedTable, restaurant]);


    const formatTimeSlot = (timeSlot) => {
        const date = new Date(timeSlot);
        const formattedDate = date.toLocaleDateString();
        const formattedTime = date.toLocaleTimeString([], { hour: '2-digit', minute: '2-digit' });
        return `${formattedDate} - ${formattedTime}`;
    };

    const handleTableSelect = (event) => {
        const selectedTableId = event.target.value;
        setSelectedTableId(selectedTableId);
    };

    const handleTimeSlotSelect = (event) => {
        const selectedTimeSlot = event.target.value;
        setSelectedTimeSlot(selectedTimeSlot);
    };

    const handleReservationConfirmation = (restaurant) => {
        const tableNumber = selectedTableId;
        const formattedTimeSlot = selectedTimeSlot;
        const restaurantId = restaurant.restaurantId;

        const encodedTableNumber = encodeURIComponent(tableNumber);
        const encodedTimeSlot = encodeURIComponent(formattedTimeSlot);
        const encodedRestaurantId = encodeURIComponent(restaurantId);

        navigate(`/reservationConfirmation/${encodedTableNumber}/${encodedTimeSlot}/${encodedRestaurantId}`);
    };


    return (
        <div className="container">
            {restaurant && (
                <>
                    <h2 className="card-title">
                        {restaurant.name} <StarRating key={restaurant.id} rating={restaurant.rating}/>
                    </h2>
                    <div className="restaurant-details">
                        <p>Operating hours: {restaurant.operatingHours}</p>
                        <p>Cuisine: {restaurant.cuisineType}</p>
                        <p>Address: {restaurant.address}</p>
                        <p>Phone: {restaurant.phone}</p>
                        <p>Website: <a href={restaurant.website}>{restaurant.website}</a></p>
                        <p>Social Media Links: {restaurant.socialMediaLinks}</p>

                        <label>Select Table:</label>
                        <select className="form-select" aria-label="Select Table" onChange={handleTableSelect}
                                value={selectedTableId}>
                            <option value="">Select Table</option>
                            {restaurant.tablesList.map((table, index) => (
                                <option key={index}
                                        value={table.id}>{`Capacity: ${table.capacity} - ${table.location}`}</option>
                            ))}
                        </select>
                        {selectedTable && (
                            <>
                                <label>Select Time Slot:</label>
                                <select className="form-select mt-2" aria-label="Select Time Slot" onChange={handleTimeSlotSelect}>
                                    <option value="">Select Time Slot</option>
                                    {filteredTimeSlots.map((timeSlot, index) => (
                                        <option key={index} value={timeSlot}>{formatTimeSlot(timeSlot)}</option>
                                    ))}
                                </select>
                            </>
                        )}
                        <br/>
                        {/* Add a button to trigger reservation confirmation */}
                        <button className="btn btn-primary" onClick={() => handleReservationConfirmation(restaurant)}>
                            Confirm Reservation
                        </button>
                    </div>
                </>
            )}
        </div>
    );
};

export default RestaurantDetails;
