import React, { useState, useEffect } from 'react';
import { useParams } from 'react-router-dom';
import axios from 'axios';
import { useNavigate } from 'react-router-dom';
import { useLocation } from 'react-router-dom';
import { jwtDecode } from "jwt-decode";
import {request} from "../axios_helper";
import restaurants from "./Restaurants";

const ReservationConfirmation = () => {
    const navigate = useNavigate();

    const location = useLocation();
    const preOrderedItems = location.state?.preOrderedItems || [];
    const [restaurant, setRestaurant] = useState({});
    const [user, setUser] = useState({});
    const [table, setTable] = useState({});
    const [reservationDateTime, setReservationDateTime] = useState('');
    const [partySize, setPartySize] = useState('');
    const [specialRequests, setSpecialRequests] = useState('');
    const { tableNumber, timeSlot, restaurantId } = useParams();

    const adjustedTimeSlot = new Date(new Date(timeSlot).getTime() + 60 * 60 * 1000).toISOString();
    useEffect(() => {
        const fetchDetails = async () => {
            try {
                const tableResponse = await axios.get(`http://localhost:8081/api/tables/${tableNumber}`);
                setTable(tableResponse.data);

                const restaurantResponse = await axios.get(`http://localhost:8081/api/restaurants/${restaurantId}`);
                setRestaurant(restaurantResponse.data);

                const token = localStorage.getItem("token");
                if (!token) {
                    console.error("No token found");
                    return;
                }
                const decodedToken = jwtDecode(token);
                const userId = decodedToken.iss;

                const userResponse = await axios.get(`http://localhost:8081/api/user/${userId}`);
                setUser(userResponse.data);
            } catch (error) {
                console.error('Error fetching table or restaurant details:', error);
            }
        };
        fetchDetails();
    }, [tableNumber, restaurantId]);

    const handleSubmit = async (e) => {
        e.preventDefault();

        const payload = {
            reservationID: 0,
            userEmail: user.email,
            rating: parseFloat(restaurant.rating) || null,
            tableNumber: parseInt(table.id, 10),
            restaurant: restaurant,
            reservationDateTime: adjustedTimeSlot,
            partySize: parseInt(partySize, 10),
            status: 'Reserved',
            specialRequests: specialRequests.trim(),
            paymentStatus: 'Pending',
            preOrderedItems: preOrderedItems.map(item => `${item.itemName}:${item.quantity}:${item.price}`)
        };


        try {
            const response = await axios.post('http://localhost:8081/api/reservations', payload);
            console.log('Reservation created successfully:', response.data);
            navigate("/reservations")
        } catch (error) {
            if (error.response) {
                alert('The selected time slot is no longer available. Please choose another time.');
            } else {
                alert('Network error. Please check your internet connection.');
            }
        }
    };

    const calculateCheckOutTime = (checkInTime) => {
        const checkIn = new Date(checkInTime);
        checkIn.setHours(checkIn.getHours() + 2);
        return checkIn.toISOString();
    };

    const initialRemainingTime = localStorage.getItem('remainingTime') || 300;
    const [remainingTime, setRemainingTime] = useState(parseInt(initialRemainingTime, 10));

    useEffect(() => {
        const timer = setInterval(() => {
            setRemainingTime((prevTime) => {
                const newTime = prevTime - 1;
                localStorage.setItem('remainingTime', newTime.toString());
                return newTime;
            });
        }, 1000);

        return () => clearInterval(timer);
    }, []);

    useEffect(() => {
        if (remainingTime <= 0) {
            localStorage.removeItem('remainingTime');
            alert("Time has expired. Please try reserving again.");
            navigate('/restaurants'); // Redirect or take necessary action
        }
    }, [remainingTime, navigate]);

    const formatTime = (timeInSeconds) => {
        const minutes = Math.floor(timeInSeconds / 60);
        const seconds = timeInSeconds % 60;
        return `${minutes.toString().padStart(2, '0')}:${seconds.toString().padStart(2, '0')}`;
    };

    const formatTimeSlot = (timeSlot) => {
        const utcDate = new Date(timeSlot);
        const localDate = new Date(utcDate.toLocaleString("en-US", { timeZone: Intl.DateTimeFormat().resolvedOptions().timeZone }));
        const formattedDate = localDate.toLocaleDateString();
        const formattedTime = localDate.toLocaleTimeString([], { hour: '2-digit', minute: '2-digit' });
        return `${formattedDate} - ${formattedTime}`;
    };

    const grandTotal = preOrderedItems.reduce((acc, item) => acc + item.price * item.quantity, 0).toFixed(2);
    const itemQuantityString = preOrderedItems
        .map(item => `${item.itemName}:${item.quantity}`)
        .join(',');

    return (
        <div className="container mt-5">
            <div className="row justify-content-center">
                <div className="col-md-6">
                    <div className="card">
                        <div className="card-header">
                            <h3 className="text-center">Reservation Confirmation</h3>
                            <p>Remaining Time: {formatTime(remainingTime)}</p>
                        </div>
                        <form onSubmit={handleSubmit}>
                            <div className="card-body">
                                <h5 className="card-title">Reservation Details</h5>
                                <p className="card-text">
                                    <strong>Restaurant:</strong> {restaurant.name || 'Loading...'} <br />
                                    <strong>Cuisine type:</strong> {restaurant.cuisineType || 'Loading...'} <br />
                                    <strong>Selected Time Slot:</strong> {formatTimeSlot(timeSlot)} <br />
                                    <strong>Party size:</strong>{' '}
                                    <input
                                        type="number"
                                        max={table.capacity}
                                        value={partySize}
                                        onChange={(e) => setPartySize(e.target.value)}
                                    />
                                    <strong>Table size:</strong> {table.capacity} <br />
                                    <strong>Special Requests:</strong>{' '}
                                    <input
                                        type="text"
                                        value={specialRequests}
                                        onChange={(e) => setSpecialRequests(e.target.value)}
                                    />
                                    <br />
                                </p>
                                <p className="card-text text-success">
                                    <strong>
                                        Check-in Time: Grace period of 15 minutes +/- the slot. For more information, call the restaurant.
                                    </strong>
                                    <br />
                                </p>
                                {preOrderedItems.length > 0 ? (
                                    <div className="row">
                                        {preOrderedItems.map((item) => (
                                            <div key={item.menuID} className="col-md-4 mb-4">
                                                <div className="list-group shadow-sm p-3">
                                                    <p className="item"><strong>Item:</strong> {item.itemName}</p>
                                                    <p className="item"><strong>Price:</strong> ${item.price.toFixed(2)}</p>
                                                    <p className="item"><strong>Quantity:</strong> {item.quantity}</p>
                                                    <p className="item"><strong>Total:</strong> ${(item.price * item.quantity).toFixed(2)}</p>
                                                </div>
                                            </div>
                                        ))}

                                        <div className="col-12 mt-4">
                                            <div className="list-group shadow-sm p-4 text-center">
                                                <h4>Grand Total: ${grandTotal}</h4>
                                            </div>
                                        </div>
                                    </div>
                                ) : (
                                    <p>No pre-ordered items.</p>
                                )}
                            </div>
                            <div className="card-footer">
                                <button type="submit" className="btn btn-primary">Submit</button>
                                <a href="/restaurants" className="btn btn-secondary mx-2">Back to Restaurants</a>
                                <a href="/" className="btn btn-secondary">Back to Home</a>
                            </div>
                        </form>
                    </div>
                </div>
            </div>
        </div>
    );
};

export default ReservationConfirmation;