<?php

namespace App\Http\Controllers\Dashboard;

use App\Helpers\Alert;
use App\Http\Requests\Dashboard\NewUserRequest;
use App\Http\Requests\Dashboard\UpdateUserData;
use App\Http\Requests\Dashboard\UpdateUserRequest;
use App\Models\Avatar;
use App\Models\Department;
use App\Models\Folder;
use Carbon\Carbon;
use Illuminate\Support\Facades\File;
use App\Models\Role;
use App\Models\User;
use App\Notifications\VerifyNewEmail;
use Illuminate\Http\Request;
use App\Notifications\WelcomeUser;
use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\Storage;

class UsersController extends Controller
{

    public function index()
    {
        return view("dashboard.users.index")->with([
            "users" => User::all(),
            "roles" => Role::all(),
            "adminAndReferents" => User::where("role_id", 1)->orWhere("role_id", 2)->get()
        ]);
    }

    public function create()
    {
        return view("dashboard.users.create")->with([
            "roles" => Role::all(),
        ]);
    }

    public function store(NewUserRequest $request)
    {
        $user = new User();

        $user->name = $request->name;
        $user->surname = $request->surname;
        $user->email = $request->email;
        $user->phone_number = $request->phone_number;
        $user->username = $request->username;
        $user->password = $user->generateTemporaryPassword();
        $user->security_code = $user->generateSecurityCode();
        $user->verify_token = $user->generateVerifyToken();

        $user->role_id = $request->userRole;

        if ($request->hasFile("avatar")) {
            $image = $request->file("avatar");
            $avatarName = $user->id . '_avatar' . time() . '.' . request()->avatar->getClientOriginalExtension();

            if (!empty($user->avatar)) {
                Storage::disk('uploads')->delete("/users/" . $user->avatar);
            }

            Storage::disk('uploads')->put("/users/" . $avatarName, File::get($image));

            $user->avatar = $avatarName;
        }

        $user->created_by = auth()->user()->id;
        $user->save();

        $user->notify(new WelcomeUser($user));

        Alert::flash("New user added successfully");

        return redirect()->route("dashboard.users.index");
    }

    public function editShow($id)
    {
        return view("dashboard.users.edit")->with([
            "user" => User::findOrFail($id),
            "roles" => Role::all(),
        ]);
    }

    public function edit(UpdateUserRequest $request, $id)
    {
        $user = User::findOrFail($id);
        $user->name = $request->name;
        $user->surname = $request->surname;
        $user->username = $request->username;
        $user->email = $request->email;
        $user->phone_number = $request->phone_number;
        $user->role_id = $request->userRole;
        $user->updated_at = Carbon::now();

        if ($request->hasFile("avatar")) {
            $image = $request->file("avatar");
            $avatarName = $user->id . '_avatar' . time() . '.' . request()->avatar->getClientOriginalExtension();

            if (!empty($user->avatar)) {
                Storage::disk('uploads')->delete("/users/" . $user->avatar);
            }

            Storage::disk('uploads')->put("/users/" . $avatarName, File::get($image));

            $user->avatar = $avatarName;
        }

        if($user->isDirty('email')) {
            $user->notify(new VerifyNewEmail($user));
        }
        $user->save();

        Alert::flash("User updated successfully");

        return redirect()->route("dashboard.users.index");
    }

    public function block(Request $request, $id)
    {
        $user = User::find($id);
        $user->is_active = false;
        $user->save();
        Alert::flash($user->name . " User blocked successfully");
        return redirect()->route("dashboard.users.index");
    }

    public function unblock(Request $request, $id)
    {
        $user = User::find($id);
        $user->is_active = true;
        $user->save();
        Alert::flash($user->name . " User unblocked successfully");
        return redirect()->route("dashboard.users.index");
    }

    public function destroy(Request $request, $id)
    {
        $user = User::find($id);

        if(Folder::where('user_id', $user->id)->count() == 0) {
            if (Storage::disk("uploads")->exists("users" . DIRECTORY_SEPARATOR . $user->avatar)) {
                Storage::disk("uploads")->delete("users" . DIRECTORY_SEPARATOR . $user->avatar);
            }

            $user->delete();

            Alert::flash($user->name . " deleted successfully");

            return redirect()->route("dashboard.users.index");
        }
        else {
            Alert::flash($user->name . "has associated folders");

            return redirect()->back();
        }
    }

    public function getUserRoles()
    {
      $roles = Role::get();
      return $roles;
    }

    public function editUserData(UpdateUserData $request, $id)
    {
        $user = User::findOrFail($id);
        $user->name = $request->name;
        $user->surname = $request->surname;
        $user->phone_number = $request->phone_number;

        if ($request->hasFile("avatar")) {
            $image = $request->file("avatar");
            $avatarName = $user->id . '_avatar' . time() . '.' . request()->avatar->getClientOriginalExtension();

            if (!empty($user->avatar)) {
                Storage::disk('uploads')->delete("/users/" . $user->avatar);
            }

            Storage::disk('uploads')->put("/users/" . $avatarName, File::get($image));

            $user->avatar = $avatarName;
        }

        $user->save();

        Alert::flash("User data updated successfully");

        return redirect()->route("dashboard.settings.index");
    }
}
