require('dotenv').config();
const Pool = require("pg").Pool;

const pool = new Pool({
    user: process.env.DB_USER,
    host: process.env.DB_HOST,
    database: process.env.DB_NAME,
    password: process.env.DB_PASSWORD,
    port: process.env.DB_PORT,
});

// Get all vehicles
const getVehicles = async () => {
    try {
        return await new Promise(function (resolve, reject) {
            pool.query(
                "SELECT vin, make, model, p_year, color, price, tax_nr, capacity, power, body FROM vehicle WHERE status = true ORDER BY p_year DESC, make, model",
                (error, results) => {
                    if (error) {
                        reject(error);
                    }
                    if (results && results.rows) {
                        resolve(results.rows);
                        console.log(`Retrieved ${results.rows.length} available vehicles`);
                    } else {
                        resolve([]);
                    }
                }
            );
        });
    } catch (error_1) {
        console.error(error_1);
        throw new Error("Internal server error");
    }
};

// Get My vehicles
const getMyVehicles = async (taxNr) => {
    try {
        return await new Promise(function (resolve, reject) {
            pool.query(
                "SELECT vin, make, model, p_year, color, price, capacity, power, body, status FROM vehicle WHERE tax_nr = $1 ORDER BY status DESC, p_year DESC, make, model",
                [taxNr],
                (error, results) => {
                    if (error) {
                        reject(error);
                    }
                    if (results && results.rows) {
                        resolve(results.rows);
                        console.log(`Retrieved ${results.rows.length} vehicles for dealership ${taxNr}`);
                    } else {
                        resolve([]);
                    }
                }
            );
        });
    } catch (error_1) {
        console.error(error_1);
        throw new Error("Internal server error");
    }
};

// Create new vehicle
const createVehicle = async (body) => {
    return new Promise(function (resolve, reject) {
        const {
            vin,
            car_body,
            car_make,
            car_model,
            p_year,
            capacity,
            power,
            color,
            price,
            tax_nr,
        } = body;
        console.log(vin, car_model, p_year);
        pool.query(
            "INSERT INTO vehicle (VIN, Body, Make, Model, P_Year, Capacity, Power, Color, Price, Status, Tax_Nr) values ($1, $2, $3, $4, $5, $6, $7, $8, $9, $10, $11) RETURNING *",
            [
                vin,
                car_body,
                car_make,
                car_model,
                p_year,
                capacity,
                power,
                color,
                price,
                true,
                tax_nr
            ],
            (error, results) => {
                if (error) {
                    reject(new Error("Not able to make new vehicle listing."));
                }
                if (results && results.rows) {
                    resolve(`The new vehicle has been entered.`);
                } else {
                    reject(new Error("Not able to make new vehicle listing."));
                }
            }
        );
    });
};

// Register new user
const registerUser = async (body) => {
    return new Promise(function (resolve, reject) {
        const { EMBG, C_name, Email, Pass, Telephone } = body;

        pool.query(
            "INSERT INTO client (EMBG, C_Name, Email, Pass, Telephone) values ($1, $2, $3, $4, $5) RETURNING *",
            [EMBG, C_name, Email, Pass, Telephone],
            (error, results) => {
                if (error) {
                    throw(error);
                }
                if (results && results.rows) {
                    resolve(`Account was created.`);
                } else {
                    throw(new Error("Not able to make new account."));
                }
            }
        );
    });
};

// Register new dealership
const registerDealership = async (body) => {
    return new Promise(function (resolve, reject) {
        const { tax_nr, d_name, brands, email, pass, director, telephones, addres } = body;

        console.log('Registering dealership with data:', { tax_nr, d_name, brands, telephones });

        // Use a client for transaction
        pool.connect((err, client, done) => {
            if (err) {
                console.error('Error getting client from pool:', err);
                reject(err);
                return;
            }

            client.query('BEGIN', (beginError) => {
                if (beginError) {
                    console.error('Begin transaction error:', beginError);
                    done();
                    reject(beginError);
                    return;
                }

                // Insert dealership without brands and telephones
                client.query(
                    "INSERT INTO dealership (Tax_Nr, D_Name, Email, Pass, Director, Addres) values ($1, $2, $3, $4, $5, $6) RETURNING *",
                    [tax_nr, d_name, email, pass, director, addres],
                    (error, results) => {
                        if (error) {
                            console.error('Dealership insert error:', error);
                            client.query('ROLLBACK', () => {
                                done();
                            });
                            reject(error);
                            return;
                        }
                        
                        if (results && results.rows) {
                            console.log('Dealership inserted successfully');
                            
                            // Insert brands if provided
                            if (brands && brands.length > 0) {
                                console.log('Inserting brands:', brands);
                                const brandInserts = brands.map(brand => {
                                    return new Promise((resolveBrand, rejectBrand) => {
                                        client.query(
                                            "INSERT INTO dealership_brands (dealership_tax_nr, brand_name) VALUES ($1, $2)",
                                            [tax_nr, brand.trim()],
                                            (brandError) => {
                                                if (brandError) {
                                                    console.error('Brand insert error:', brandError);
                                                    rejectBrand(brandError);
                                                } else {
                                                    resolveBrand();
                                                }
                                            }
                                        );
                                    });
                                });
                                
                                Promise.all(brandInserts)
                                    .then(() => {
                                        console.log('All brands inserted successfully');
                                        // Insert telephones if provided
                                        if (telephones && telephones.length > 0) {
                                            console.log('Inserting telephones:', telephones);
                                            const phoneInserts = telephones.map(phone => {
                                                return new Promise((resolvePhone, rejectPhone) => {
                                                    client.query(
                                                        "INSERT INTO dealership_telephones (dealership_tax_nr, phone_number) VALUES ($1, $2)",
                                                        [tax_nr, phone.trim()],
                                                        (phoneError) => {
                                                            if (phoneError) {
                                                                console.error('Phone insert error:', phoneError);
                                                                rejectPhone(phoneError);
                                                            } else {
                                                                resolvePhone();
                                                            }
                                                        }
                                                    );
                                                });
                                            });
                                            
                                            Promise.all(phoneInserts)
                                                .then(() => {
                                                    console.log('All telephones inserted successfully');
                                                    client.query('COMMIT', (commitError) => {
                                                        done();
                                                        if (commitError) {
                                                            console.error('Commit error:', commitError);
                                                            reject(commitError);
                                                        } else {
                                                            console.log('Transaction committed successfully');
                                                            resolve(`Dealership account was created.`);
                                                        }
                                                    });
                                                })
                                                .catch((phoneError) => {
                                                    console.error('Phone insertion failed:', phoneError);
                                                    client.query('ROLLBACK', () => {
                                                        done();
                                                    });
                                                    reject(phoneError);
                                                });
                                        } else {
                                            client.query('COMMIT', (commitError) => {
                                                done();
                                                if (commitError) {
                                                    console.error('Commit error:', commitError);
                                                    reject(commitError);
                                                } else {
                                                    console.log('Transaction committed successfully (no phones)');
                                                    resolve(`Dealership account was created.`);
                                                }
                                            });
                                        }
                                    })
                                    .catch((brandError) => {
                                        console.error('Brand insertion failed:', brandError);
                                        client.query('ROLLBACK', () => {
                                            done();
                                        });
                                        reject(brandError);
                                    });
                            } else {
                                // No brands, just insert telephones if provided
                                if (telephones && telephones.length > 0) {
                                    console.log('Inserting telephones (no brands):', telephones);
                                    const phoneInserts = telephones.map(phone => {
                                        return new Promise((resolvePhone, rejectPhone) => {
                                            client.query(
                                                "INSERT INTO dealership_telephones (dealership_tax_nr, phone_number) VALUES ($1, $2)",
                                                [tax_nr, phone.trim()],
                                                (phoneError) => {
                                                    if (phoneError) {
                                                        console.error('Phone insert error:', phoneError);
                                                        rejectPhone(phoneError);
                                                    } else {
                                                        resolvePhone();
                                                    }
                                                }
                                            );
                                        });
                                    });
                                    
                                    Promise.all(phoneInserts)
                                        .then(() => {
                                            console.log('All telephones inserted successfully (no brands)');
                                            client.query('COMMIT', (commitError) => {
                                                done();
                                                if (commitError) {
                                                    console.error('Commit error:', commitError);
                                                    reject(commitError);
                                                } else {
                                                    console.log('Transaction committed successfully (no brands)');
                                                    resolve(`Dealership account was created.`);
                                                }
                                            });
                                        })
                                        .catch((phoneError) => {
                                            console.error('Phone insertion failed (no brands):', phoneError);
                                            client.query('ROLLBACK', () => {
                                                done();
                                            });
                                            reject(phoneError);
                                        });
                                } else {
                                    client.query('COMMIT', (commitError) => {
                                        done();
                                        if (commitError) {
                                            console.error('Commit error:', commitError);
                                            reject(commitError);
                                        } else {
                                            console.log('Transaction committed successfully (no brands or phones)');
                                            resolve(`Dealership account was created.`);
                                        }
                                    });
                                }
                            }
                        } else {
                            client.query('ROLLBACK', () => {
                                done();
                            });
                            reject(new Error("Not able to make new dealership account."));
                        }
                    }
                );
            });
        });
    });
};

// Get single agreement by VIN
const getAgreement = async (slug) => {
    console.log(slug);
    try {
        return await new Promise(function (resolve, reject) {
            pool.query(
                `SELECT a.*, v.make, v.model, v.p_year, v.color,
                        c.c_name, c.embg, d.d_name
                 FROM agreement a
                 JOIN vehicle v ON a.vin = v.vin
                 JOIN client c ON a.embg = c.embg
                 JOIN dealership d ON a.tax_nr = d.tax_nr
                 WHERE a.vin = $1`,
                [slug],
                (error, results) => {
                    if (error) {
                        reject(error);
                    }
                    if (results && results.rows) {
                        resolve(results.rows);
                    } else {
                        reject(new Error("No results found"));
                    }
                }
            );
        });
    } catch (error_1) {
        console.error(error_1);
        throw new Error("Internal server error");
    }
};

// Get single agreement by agreement ID
const getAgreementById = async (agreementId) => {
    try {
        return await new Promise(function (resolve, reject) {
            pool.query(
                `SELECT a.*, v.make, v.model, v.p_year, v.color,
                        c.c_name, c.embg, d.d_name
                 FROM agreement a
                 JOIN vehicle v ON a.vin = v.vin
                 JOIN client c ON a.embg = c.embg
                 JOIN dealership d ON a.tax_nr = d.tax_nr
                 WHERE a.a_id = $1`,
                [agreementId],
                (error, results) => {
                    if (error) {
                        reject(error);
                    }
                    if (results && results.rows && results.rows.length > 0) {
                        resolve(results.rows[0]);
                    } else {
                        resolve(null);
                    }
                }
            );
        });
    } catch (error_1) {
        console.error(error_1);
        throw new Error("Internal server error");
    }
};

// Get all agreements with payment status for a dealership
const getDealershipAgreementsWithPaymentStatus = async (taxNr) => {
    try {
        return await new Promise(function (resolve, reject) {
            pool.query(
                `SELECT a.a_id, a.price, a.status, a.datum, a.vin, a.embg, a.tax_nr,
                        v.make, v.model, v.p_year, v.color,
                        c.c_name, c.embg, d.d_name,
                        CASE WHEN p.p_id IS NOT NULL THEN true ELSE false END as payment_exists,
                        p.p_id, p.amount as payment_amount, p.bank, p.iban
                 FROM agreement a
                 JOIN vehicle v ON a.vin = v.vin
                 JOIN client c ON a.embg = c.embg
                 JOIN dealership d ON a.tax_nr = d.tax_nr
                 LEFT JOIN payment p ON a.a_id = p.a_id
                 WHERE a.tax_nr = $1
                 ORDER BY a.datum DESC`,
                [taxNr],
                (error, results) => {
                    if (error) {
                        reject(error);
                    }
                    if (results && results.rows) {
                        resolve(results.rows);
                        console.log(`Retrieved ${results.rows.length} agreements for dealership ${taxNr}`);
                    } else {
                        resolve([]);
                    }
                }
            );
        });
    } catch (error_1) {
        console.error(error_1);
        throw new Error("Internal server error");
    }
};

// Delete a vehicle
const deleteVehicle = async (vin, taxNr) => {
    try {
        return await new Promise(function (resolve, reject) {
            // First check if the vehicle belongs to the dealership and is not sold
            pool.query(
                `SELECT vin, status FROM vehicle WHERE vin = $1 AND tax_nr = $2`,
                [vin, taxNr],
                (error, results) => {
                    if (error) {
                        reject(error);
                        return;
                    }
                    
                    if (!results.rows || results.rows.length === 0) {
                        reject(new Error("Vehicle not found or you don't have permission to delete it"));
                        return;
                    }
                    
                    const vehicle = results.rows[0];
                    if (vehicle.status === false) {
                        reject(new Error("Cannot delete sold vehicles"));
                        return;
                    }
                    
                    // Delete the vehicle
                    pool.query(
                        `DELETE FROM vehicle WHERE vin = $1 AND tax_nr = $2`,
                        [vin, taxNr],
                        (deleteError, deleteResults) => {
                            if (deleteError) {
                                reject(deleteError);
                            } else {
                                resolve({ success: true, message: "Vehicle deleted successfully" });
                            }
                        }
                    );
                }
            );
        });
    } catch (error_1) {
        console.error(error_1);
        throw new Error("Internal server error");
    }
};

// Login dealership
const loginDealership = async (email, password) => {
    try {
        return await new Promise(function (resolve, reject) {
            pool.query(
                "SELECT * FROM dealership WHERE email = $1 AND pass = $2",
                [email, password],
                (error, results) => {
                    if (error) {
                        reject(error);
                        return;
                    }
                    if (results && results.rows && results.rows.length > 0) {
                        const dealership = results.rows[0];
                        resolve(dealership);
                    } else {
                        resolve(null);
                    }
                }
            );
        });
    } catch (error_1) {
        console.error(error_1);
        throw new Error("Internal server error");
    }
};

// Login client
const loginClient = async (email, password) => {
    try {
        return await new Promise(function (resolve, reject) {
            pool.query(
                "SELECT embg, c_name, email, pass, telephone FROM client WHERE email = $1 AND pass = $2",
                [email, password],
                (error, results) => {
                    if (error) {
                        reject(error);
                    }
                    if (results && results.rows && results.rows.length > 0) {
                        resolve(results.rows[0]);
                    } else {
                        resolve(null);
                    }
                }
            );
        });
    } catch (error_1) {
        console.error(error_1);
        throw new Error("Internal server error");
    }
};

// Create agreement
const createAgreement = async (body) => {
    return new Promise(function (resolve, reject) {
        const {
            A_Id,
            Price,
            Status,
            Datum,
            VIN,
            EMBG,
            Tax_Nr
        } = body;
        
        pool.query(
            "INSERT INTO agreement (A_Id, Price, Status, Datum, Tax_Nr, VIN, EMBG) VALUES ($1, $2, $3, $4, $5, $6, $7) RETURNING *",
            [A_Id, Price, Status, Datum, Tax_Nr, VIN, EMBG],
            (error, results) => {
                if (error) {
                    reject(new Error("Not able to create agreement."));
                }
                if (results && results.rows) {
                    // Update vehicle status to sold
                    pool.query(
                        "UPDATE vehicle SET status = false WHERE vin = $1",
                        [VIN],
                        (updateError, updateResults) => {
                            if (updateError) {
                                console.error("Error updating vehicle status:", updateError);
                            }
                        }
                    );
                    resolve(`Agreement created successfully.`);
                } else {
                    reject(new Error("Not able to create agreement."));
                }
            }
        );
    });
};

// Get client agreements
const getClientAgreements = async (embg) => {
    try {
        return await new Promise(function (resolve, reject) {
            pool.query(
                `SELECT a.a_id, a.price, a.status, a.datum, a.vin, a.embg, a.tax_nr,
                        v.make, v.model, v.p_year, v.color, v.price as vehicle_price, 
                        d.d_name as dealership_name,
                        CASE WHEN p.p_id IS NOT NULL THEN true ELSE false END as payment_exists
                 FROM agreement a 
                 JOIN vehicle v ON a.vin = v.vin 
                 JOIN dealership d ON a.tax_nr = d.tax_nr 
                 LEFT JOIN payment p ON a.a_id = p.a_id
                 WHERE a.embg = $1
                 ORDER BY a.datum DESC`,
                [embg],
                (error, results) => {
                    if (error) {
                        reject(error);
                    }
                    if (results && results.rows) {
                        resolve(results.rows);
                        console.log(`Retrieved ${results.rows.length} agreements for client ${embg}`);
                    } else {
                        resolve([]);
                    }
                }
            );
        });
    } catch (error_1) {
        console.error(error_1);
        throw new Error("Internal server error");
    }
};

// Create payment
const createPayment = async (body) => {
    return new Promise(function (resolve, reject) {
        const {
            Bank,
            IBAN,
            Amount,
            A_Id,
            EMBG
        } = body;
        
        pool.query(
            "INSERT INTO payment (Bank, IBAN, Amount, EMBG, A_Id) VALUES ($1, $2, $3, $4, $5) RETURNING *",
            [Bank, IBAN, Amount, EMBG, A_Id],
            (error, results) => {
                if (error) {
                    reject(new Error("Not able to create payment."));
                }
                if (results && results.rows) {
                    resolve(`Payment created successfully.`);
                } else {
                    reject(new Error("Not able to create payment."));
                }
            }
        );
    });
};

// Check if agreement has payment
const checkAgreementPayment = async (aId) => {
    try {
        return await new Promise(function (resolve, reject) {
            pool.query(
                "SELECT * FROM payment WHERE a_id = $1",
                [aId],
                (error, results) => {
                    if (error) {
                        reject(error);
                    }
                    if (results && results.rows && results.rows.length > 0) {
                        resolve(true);
                    } else {
                        resolve(false);
                    }
                }
            );
        });
    } catch (error_1) {
        console.error(error_1);
        throw new Error("Internal server error");
    }
};

// Get payment details for an agreement
const getPaymentDetails = async (aId) => {
    try {
        return await new Promise(function (resolve, reject) {
            pool.query(
                `SELECT p.*, a.price as agreement_price, a.datum as agreement_date,
                        v.vin, v.make, v.model, v.p_year, v.color,
                        d.d_name as dealership_name,
                        c.c_name as client_name
                 FROM payment p
                 JOIN agreement a ON p.a_id = a.a_id
                 JOIN vehicle v ON a.vin = v.vin
                 JOIN dealership d ON a.tax_nr = d.tax_nr
                 JOIN client c ON a.embg = c.embg
                 WHERE p.a_id = $1`,
                [aId],
                (error, results) => {
                    if (error) {
                        reject(error);
                    }
                    if (results && results.rows && results.rows.length > 0) {
                        resolve(results.rows[0]);
                    } else {
                        resolve(null);
                    }
                }
            );
        });
    } catch (error_1) {
        console.error(error_1);
        throw new Error("Internal server error");
    }
};

// Get payment details by VIN
const getPaymentDetailsByVin = async (vin) => {
    try {
        return await new Promise(function (resolve, reject) {
            pool.query(
                `SELECT p.*, a.price as agreement_price, a.datum as agreement_date,
                        v.vin, v.make, v.model, v.p_year, v.color,
                        d.d_name as dealership_name,
                        c.c_name as client_name
                 FROM payment p
                 JOIN agreement a ON p.a_id = a.a_id
                 JOIN vehicle v ON a.vin = v.vin
                 JOIN dealership d ON a.tax_nr = d.tax_nr
                 JOIN client c ON a.embg = c.embg
                 WHERE v.vin = $1`,
                [vin],
                (error, results) => {
                    if (error) {
                        reject(error);
                    }
                    if (results && results.rows && results.rows.length > 0) {
                        resolve(results.rows[0]);
                    } else {
                        resolve(null);
                    }
                }
            );
        });
    } catch (error_1) {
        console.error(error_1);
        throw new Error("Internal server error");
    }
};

// Get payments for a dealership (all payments for their agreements)
const getDealershipPayments = async (taxNr) => {
    try {
        return await new Promise(function (resolve, reject) {
            pool.query(
                `SELECT p.*, a.price as agreement_price, a.datum as agreement_date,
                        v.make, v.model, v.p_year, v.color,
                        c.c_name as client_name, c.embg as client_embg
                 FROM payment p
                 JOIN agreement a ON p.a_id = a.a_id
                 JOIN vehicle v ON a.vin = v.vin
                 JOIN client c ON a.embg = c.embg
                 WHERE a.tax_nr = $1
                 ORDER BY p.p_id DESC`,
                [taxNr],
                (error, results) => {
                    if (error) {
                        reject(error);
                    }
                    if (results && results.rows) {
                        resolve(results.rows);
                    } else {
                        reject(new Error("No results found"));
                    }
                }
            );
        });
    } catch (error_1) {
        console.error(error_1);
        throw new Error("Internal server error");
    }
};

// Get all agreements for a dealership
const getDealershipAgreements = async (taxNr) => {
    try {
        return await new Promise(function (resolve, reject) {
            pool.query(
                `SELECT a.*, v.make, v.model, v.p_year, v.color,
                        c.c_name, c.embg, d.d_name
                 FROM agreement a
                 JOIN vehicle v ON a.vin = v.vin
                 JOIN client c ON a.embg = c.embg
                 JOIN dealership d ON a.tax_nr = d.tax_nr
                 WHERE a.tax_nr = $1
                 ORDER BY a.datum DESC`,
                [taxNr],
                (error, results) => {
                    if (error) {
                        reject(error);
                    }
                    if (results && results.rows) {
                        resolve(results.rows);
                    } else {
                        reject(new Error("No results found"));
                    }
                }
            );
        });
    } catch (error_1) {
        console.error(error_1);
        throw new Error("Internal server error");
    }
};

// Get dealership brands
const getDealershipBrands = async (taxNr) => {
    try {
        return await new Promise(function (resolve, reject) {
            pool.query(
                "SELECT brand_name FROM dealership_brands WHERE dealership_tax_nr = $1",
                [taxNr],
                (error, results) => {
                    if (error) {
                        reject(error);
                    }
                    if (results && results.rows) {
                        resolve(results.rows.map(row => row.brand_name));
                    } else {
                        resolve([]);
                    }
                }
            );
        });
    } catch (error_1) {
        console.error(error_1);
        throw new Error("Internal server error");
    }
};

// Get dealership telephones
const getDealershipTelephones = async (taxNr) => {
    try {
        return await new Promise(function (resolve, reject) {
            pool.query(
                "SELECT phone_number FROM dealership_telephones WHERE dealership_tax_nr = $1",
                [taxNr],
                (error, results) => {
                    if (error) {
                        reject(error);
                    }
                    if (results && results.rows) {
                        resolve(results.rows.map(row => row.phone_number));
                    } else {
                        resolve([]);
                    }
                }
            );
        });
    } catch (error_1) {
        console.error(error_1);
        throw new Error("Internal server error");
    }
};

// Get dealership with brands and telephones
const getDealershipWithDetails = async (taxNr) => {
    try {
        return await new Promise(function (resolve, reject) {
            pool.query(
                "SELECT * FROM dealership WHERE tax_nr = $1",
                [taxNr],
                async (error, results) => {
                    if (error) {
                        reject(error);
                        return;
                    }
                    if (results && results.rows && results.rows.length > 0) {
                        const dealership = results.rows[0];
                        try {
                            const brands = await getDealershipBrands(taxNr);
                            const telephones = await getDealershipTelephones(taxNr);
                            resolve({
                                ...dealership,
                                brands: brands,
                                telephones: telephones
                            });
                        } catch (detailError) {
                            reject(detailError);
                        }
                    } else {
                        resolve(null);
                    }
                }
            );
        });
    } catch (error_1) {
        console.error(error_1);
        throw new Error("Internal server error");
    }
};

// Search vehicles by make, model, year, price range
const searchVehicles = async (searchParams) => {
    try {
        return await new Promise(function (resolve, reject) {
            const { make, model, minYear, maxYear, minPrice, maxPrice, limit = 50 } = searchParams;
            
            let query = "SELECT vin, make, model, p_year, color, price, tax_nr, capacity, power, body FROM vehicle WHERE status = true";
            const params = [];
            let paramCount = 0;
            
            if (make) {
                paramCount++;
                query += ` AND make ILIKE $${paramCount}`;
                params.push(`%${make}%`);
            }
            
            if (model) {
                paramCount++;
                query += ` AND model ILIKE $${paramCount}`;
                params.push(`%${model}%`);
            }
            
            if (minYear) {
                paramCount++;
                query += ` AND p_year >= $${paramCount}`;
                params.push(minYear);
            }
            
            if (maxYear) {
                paramCount++;
                query += ` AND p_year <= $${paramCount}`;
                params.push(maxYear);
            }
            
            if (minPrice) {
                paramCount++;
                query += ` AND price >= $${paramCount}`;
                params.push(minPrice);
            }
            
            if (maxPrice) {
                paramCount++;
                query += ` AND price <= $${paramCount}`;
                params.push(maxPrice);
            }
            
            query += ` ORDER BY p_year DESC, make, model LIMIT $${paramCount + 1}`;
            params.push(limit);
            
            pool.query(query, params, (error, results) => {
                if (error) {
                    reject(error);
                } else {
                    resolve(results.rows);
                    console.log(`Search returned ${results.rows.length} vehicles`);
                }
            });
        });
    } catch (error_1) {
        console.error(error_1);
        throw new Error("Internal server error");
    }
};

// Get vehicles by price range
const getVehiclesByPriceRange = async (minPrice, maxPrice) => {
    try {
        return await new Promise(function (resolve, reject) {
            pool.query(
                "SELECT vin, make, model, p_year, color, price, tax_nr, capacity, power, body FROM vehicle WHERE status = true AND price BETWEEN $1 AND $2 ORDER BY price ASC, p_year DESC",
                [minPrice, maxPrice],
                (error, results) => {
                    if (error) {
                        reject(error);
                    } else {
                        resolve(results.rows);
                        console.log(`Found ${results.rows.length} vehicles in price range ${minPrice}-${maxPrice}`);
                    }
                }
            );
        });
    } catch (error_1) {
        console.error(error_1);
        throw new Error("Internal server error");
    }
};

// Get vehicles by year range
const getVehiclesByYearRange = async (minYear, maxYear) => {
    try {
        return await new Promise(function (resolve, reject) {
            pool.query(
                "SELECT vin, make, model, p_year, color, price, tax_nr, capacity, power, body FROM vehicle WHERE status = true AND p_year BETWEEN $1 AND $2 ORDER BY p_year DESC, make, model",
                [minYear, maxYear],
                (error, results) => {
                    if (error) {
                        reject(error);
                    } else {
                        resolve(results.rows);
                        console.log(`Found ${results.rows.length} vehicles in year range ${minYear}-${maxYear}`);
                    }
                }
            );
        });
    } catch (error_1) {
        console.error(error_1);
        throw new Error("Internal server error");
    }
};

// Get all dealerships (for debugging)
const getAllDealerships = async () => {
    try {
        return await new Promise(function (resolve, reject) {
            pool.query(
                "SELECT tax_nr, d_name, email, pass FROM dealership",
                (error, results) => {
                    if (error) {
                        reject(error);
                    }
                    if (results && results.rows) {
                        resolve(results.rows);
                    } else {
                        reject(new Error("No results found"));
                    }
                }
            );
        });
    } catch (error_1) {
        console.error(error_1);
        throw new Error("Internal server error");
    }
};

module.exports = {
    getVehicles,
    getMyVehicles,
    createVehicle,
    deleteVehicle,
    registerUser,
    registerDealership,
    getAgreement,
    getAgreementById,
    getDealershipAgreementsWithPaymentStatus,
    loginDealership,
    loginClient,
    createAgreement,
    getClientAgreements,
    createPayment,
    checkAgreementPayment,
    getPaymentDetails,
    getPaymentDetailsByVin,
    getDealershipPayments,
    getDealershipAgreements,
    getAllDealerships,
    getDealershipBrands,
    getDealershipTelephones,
    getDealershipWithDetails,
    searchVehicles,
    getVehiclesByPriceRange,
    getVehiclesByYearRange,
};
