const express = require("express");
var cors = require("cors");
const session = require("express-session");
const app = express();
const port = 3001;

// Simple in-memory store for sessions
const sessions = {};

app.use(express.json());
app.use(session({
    secret: 'carzone-secret-key',
    resave: true,
    saveUninitialized: true,
    cookie: { secure: false, maxAge: 24 * 60 * 60 * 1000 } // 24 hours
}));

app.use(function (req, res, next) {
    res.setHeader("Access-Control-Allow-Origin", "http://localhost:3000");
    res.setHeader(
        "Access-Control-Allow-Methods",
        "GET,POST,PUT,DELETE,OPTIONS"
    );
    res.setHeader(
        "Access-Control-Allow-Headers",
        "Content-Type, Access-Control-Allow-Headers"
    );
    res.setHeader("Access-Control-Allow-Credentials", "true");
    next();
});

//connection functions
const connection_model = require("./connectionModel");

// Simple authentication middleware
const requireAuth = (req, res, next) => {
    const sessionId = req.sessionID;
    const userSession = sessions[sessionId];
    
    if (userSession && userSession.userId && userSession.userType) {
        req.user = userSession;
        next();
    } else {
        res.status(401).json({ error: 'Unauthorized' });
    }
};

const requireDealership = (req, res, next) => {
    const sessionId = req.sessionID;
    const userSession = sessions[sessionId];
    
    if (userSession && userSession.userType === 'dealership') {
        req.user = userSession;
        next();
    } else {
        res.status(401).json({ error: 'Unauthorized - Dealership access required' });
    }
};

const requireClient = (req, res, next) => {
    const sessionId = req.sessionID;
    const userSession = sessions[sessionId];
    
    if (userSession && userSession.userType === 'client') {
        req.user = userSession;
        next();
    } else {
        res.status(401).json({ error: 'Unauthorized - Client access required' });
    }
};

//Server requests start here

// For home page all listings
app.get("/", (req, res) => {
    connection_model
        .getVehicles()
        .then((response) => {
            res.status(200).send(response);
        })
        .catch((error) => {
            res.status(500).send(error);
        });
});

// Authentication endpoints
app.post("/login/dealership", (req, res) => {
    const { email, pass } = req.body;

    connection_model
        .loginDealership(email, pass)
        .then((response) => {
            if (response) {
                // Store session data in our in-memory store
                const sessionId = req.sessionID;
                sessions[sessionId] = {
                    userId: response.tax_nr,
                    userType: 'dealership',
                    userName: response.d_name
                };
                
                res.status(200).json({ success: true, user: response });
            } else {

                res.status(401).json({ success: false, message: 'Invalid credentials' });
            }
        })
        .catch((error) => {
            res.status(500).json({ success: false, message: 'Server error' });
        });
});

app.post("/login/client", (req, res) => {
    const { email, pass } = req.body;
    connection_model
        .loginClient(email, pass)
        .then((response) => {
            if (response) {
                // Store session data in our in-memory store
                const sessionId = req.sessionID;
                sessions[sessionId] = {
                    userId: response.embg,
                    userType: 'client',
                    userName: response.c_name
                };
                
                res.status(200).json({ success: true, user: response });
            } else {
                res.status(401).json({ success: false, message: 'Invalid credentials' });
            }
        })
        .catch((error) => {
            res.status(500).json({ success: false, message: 'Server error' });
        });
});

app.post("/logout", (req, res) => {
    const sessionId = req.sessionID;
    delete sessions[sessionId];
    req.session.destroy((err) => {
        if (err) {
            res.status(500).json({ success: false, message: 'Could not log out' });
        } else {
            res.status(200).json({ success: true, message: 'Logged out successfully' });
        }
    });
});

app.get("/auth/status", (req, res) => {
    const sessionId = req.sessionID;
    const userSession = sessions[sessionId];
    
    if (userSession && userSession.userId && userSession.userType) {
        res.status(200).json({
            loggedIn: true,
            userId: userSession.userId,
            userType: userSession.userType,
            userName: userSession.userName
        });
    } else {
        res.status(200).json({ loggedIn: false });
    }
});



// For My list - now requires dealership authentication
app.get("/mylist", requireDealership, (req, res) => {
    connection_model
        .getMyVehicles(req.user.userId)
        .then((response) => {
            res.status(200).send(response);
        })
        .catch((error) => {
            res.status(500).send(error);
        });
});

// Create agreement - requires dealership authentication
app.post("/agreement", requireDealership, (req, res) => {
    const agreementData = {
        ...req.body,
        Tax_Nr: req.user.userId
    };
    connection_model
        .createAgreement(agreementData)
        .then((response) => {
            res.status(200).send(response);
        })
        .catch((error) => {
            res.status(500).send(error);
        });
});

// Get client agreements - requires client authentication
app.get("/client/agreements", requireClient, (req, res) => {
    connection_model
        .getClientAgreements(req.user.userId)
        .then((response) => {
            res.status(200).send(response);
        })
        .catch((error) => {
            res.status(500).send(error);
        });
});

// Create payment - requires client authentication
app.post("/payment", requireClient, (req, res) => {
    const paymentData = {
        ...req.body,
        EMBG: req.user.userId
    };
    connection_model
        .createPayment(paymentData)
        .then((response) => {
            res.status(200).send(response);
        })
        .catch((error) => {
            res.status(500).send(error);
        });
});

// For create vehicle POST method - requires dealership authentication
app.post("/createvehicle", requireDealership, (req, res) => {
    const vehicleData = {
        ...req.body,
        tax_nr: req.user.userId
    };
    connection_model
        .createVehicle(vehicleData)
        .then((response) => {
            res.status(200).send(response);
        })
        .catch((error) => {
            res.status(500).send(error);
        });
});

// For register new user POST
app.post("/registeruser", (req, res) => {
    connection_model
        .registerUser(req.body)
        .then((response) => {
            res.status(200).send(response);
        })
        .catch((error) => {
            res.status(500).send(error);
        });
});

// For register new dealership POST
app.post("/registerdealership", (req, res) => {
    connection_model
        .registerDealership(req.body)
        .then((response) => {
            res.status(200).send(response);
        })
        .catch((error) => {
            res.status(500).send(error);
        });
});

// Get payment details for an agreement
app.get("/payment/:agreementId", (req, res) => {
    const { agreementId } = req.params;
    connection_model
        .getPaymentDetails(agreementId)
        .then((response) => {
            if (response) {
                res.status(200).json(response);
            } else {
                res.status(404).json({ error: 'Payment not found' });
            }
        })
        .catch((error) => {
            res.status(500).json({ error: error.message });
        });
});

// Get payment details by VIN
app.get("/payment/vin/:vin", (req, res) => {
    const { vin } = req.params;
    connection_model
        .getPaymentDetailsByVin(vin)
        .then((response) => {
            if (response) {
                res.status(200).json(response);
            } else {
                res.status(404).json({ error: 'Payment not found' });
            }
        })
        .catch((error) => {
            res.status(500).json({ error: error.message });
        });
});

// Get all payments for a dealership
app.get("/dealership/payments", requireDealership, (req, res) => {
    connection_model
        .getDealershipPayments(req.user.userId)
        .then((response) => {
            res.status(200).json(response);
        })
        .catch((error) => {
            res.status(500).json({ error: error.message });
        });
});

// Get all agreements for a dealership
app.get("/dealership/agreements", requireDealership, (req, res) => {
    connection_model
        .getDealershipAgreements(req.user.userId)
        .then((response) => {
            res.status(200).json(response);
        })
        .catch((error) => {
            res.status(500).json({ error: error.message });
        });
});

// Get all agreements with payment status for a dealership
app.get("/dealership/agreements-with-payments", requireDealership, (req, res) => {
    connection_model
        .getDealershipAgreementsWithPaymentStatus(req.user.userId)
        .then((response) => {
            res.status(200).json(response);
        })
        .catch((error) => {
            res.status(500).json({ error: error.message });
        });
});

// Delete a vehicle
app.delete("/vehicle/:vin", requireDealership, (req, res) => {
    const { vin } = req.params;
    const taxNr = req.user.userId;
    
    connection_model
        .deleteVehicle(vin, taxNr)
        .then((response) => {
            res.status(200).json(response);
        })
        .catch((error) => {
            res.status(400).json({ error: error.message });
        });
});

// To get single agreement based on slug passed
app.get("/agreement/:slug", (req, res) => {
    connection_model
        .getAgreement(req.params.slug)
        .then((response) => {
            res.status(200).send(response);
        })
        .catch((error) => {
            res.status(500).send(error);
        });
});

// Get agreement by agreement ID
app.get("/agreement/id/:agreementId", (req, res) => {
    connection_model
        .getAgreementById(req.params.agreementId)
        .then((response) => {
            if (response) {
                res.status(200).json(response);
            } else {
                res.status(404).json({ error: 'Agreement not found' });
            }
        })
        .catch((error) => {
            res.status(500).json({ error: error.message });
        });
});

// Get dealership brands
app.get("/dealership/brands", requireDealership, (req, res) => {
    connection_model
        .getDealershipBrands(req.user.userId)
        .then((response) => {
            res.status(200).json(response);
        })
        .catch((error) => {
            res.status(500).json({ error: error.message });
        });
});

// Get dealership telephones
app.get("/dealership/telephones", requireDealership, (req, res) => {
    connection_model
        .getDealershipTelephones(req.user.userId)
        .then((response) => {
            res.status(200).json(response);
        })
        .catch((error) => {
            res.status(500).json({ error: error.message });
        });
});

// Get dealership with complete details (brands and telephones)
app.get("/dealership/details", requireDealership, (req, res) => {
    connection_model
        .getDealershipWithDetails(req.user.userId)
        .then((response) => {
            if (response) {
                res.status(200).json(response);
            } else {
                res.status(404).json({ error: 'Dealership not found' });
            }
        })
        .catch((error) => {
            res.status(500).json({ error: error.message });
        });
});

// Search vehicles with filters
app.get("/vehicles/search", (req, res) => {
    const searchParams = {
        make: req.query.make,
        model: req.query.model,
        minYear: req.query.minYear ? parseInt(req.query.minYear) : null,
        maxYear: req.query.maxYear ? parseInt(req.query.maxYear) : null,
        minPrice: req.query.minPrice ? parseFloat(req.query.minPrice) : null,
        maxPrice: req.query.maxPrice ? parseFloat(req.query.maxPrice) : null,
        limit: req.query.limit ? parseInt(req.query.limit) : 50
    };

    connection_model
        .searchVehicles(searchParams)
        .then((response) => {
            res.status(200).json(response);
        })
        .catch((error) => {
            res.status(500).json({ error: error.message });
        });
});

// Get vehicles by price range
app.get("/vehicles/price-range", (req, res) => {
    const { minPrice, maxPrice } = req.query;
    
    if (!minPrice || !maxPrice) {
        return res.status(400).json({ error: 'minPrice and maxPrice are required' });
    }

    connection_model
        .getVehiclesByPriceRange(parseFloat(minPrice), parseFloat(maxPrice))
        .then((response) => {
            res.status(200).json(response);
        })
        .catch((error) => {
            res.status(500).json({ error: error.message });
        });
});

// Get vehicles by year range
app.get("/vehicles/year-range", (req, res) => {
    const { minYear, maxYear } = req.query;
    
    if (!minYear || !maxYear) {
        return res.status(400).json({ error: 'minYear and maxYear are required' });
    }

    connection_model
        .getVehiclesByYearRange(parseInt(minYear), parseInt(maxYear))
        .then((response) => {
            res.status(200).json(response);
        })
        .catch((error) => {
            res.status(500).json({ error: error.message });
        });
});

app.listen(port, () => {
    console.log(`App running on port ${port}.`);
});
