import { useState, useEffect } from "react";
import { Button, Flex, SimpleGrid, Text, Box, Badge, useToast } from "@chakra-ui/react";
import { useNavigate } from "react-router-dom";

const Reports = () => {
    const [agreements, setAgreements] = useState([]);
    const [loading, setLoading] = useState(true);
    const navigate = useNavigate();
    const toast = useToast();

    useEffect(() => {
        fetchAllAgreements();
    }, []);

    const fetchAllAgreements = async () => {
        setLoading(true);
        try {
            const response = await fetch("http://localhost:3001/dealership/agreements-with-payments", {
                credentials: 'include'
            });

            if (response.ok) {
                const data = await response.json();
                setAgreements(data);
            } else if (response.status === 401) {
                toast({
                    title: "Unauthorized",
                    description: "Please log in as a dealership to view agreements",
                    status: "error",
                    duration: 3000,
                    isClosable: true,
                });
                navigate("/dealershiplogin");
            } else {
                throw new Error("Failed to fetch agreements");
            }
        } catch (error) {
            toast({
                title: "Error",
                description: "Failed to load agreements",
                status: "error",
                duration: 3000,
                isClosable: true,
            });
        } finally {
            setLoading(false);
        }
    };

    const handleViewDocuments = (agreement) => {
        navigate("/documents", { state: { agreementData: agreement, vin: agreement.vin } });
    };

    const AgreementCard = ({ agreement }) => {
        return (
            <Box
                borderWidth="1px"
                borderRadius="lg"
                overflow="hidden"
                p="4"
                bg={agreement.payment_exists ? "green.50" : "white"}
                boxShadow="md"
                _hover={{ boxShadow: "lg" }}
                transition="all 0.3s"
                borderColor={agreement.payment_exists ? "green.200" : "gray.200"}>
                
                <Flex direction="column" gap="2">
                    <Flex justify="space-between" align="center">
                        <Text fontWeight="bold" fontSize="lg">
                            {agreement.make} {agreement.model}
                        </Text>
                        <Badge 
                            colorScheme={agreement.payment_exists ? "green" : "orange"}
                            fontSize="sm">
                            {agreement.payment_exists ? "PAID" : "PENDING"}
                        </Badge>
                    </Flex>
                    
                    <Text fontSize="sm" color="gray.600">
                        Year: {agreement.p_year}
                    </Text>
                    
                    <Text fontSize="sm" color="gray.600">
                        Color: {agreement.color}
                    </Text>
                    
                    <Text fontSize="sm" color="gray.600">
                        Client: {agreement.c_name}
                    </Text>
                    
                    <Text fontSize="sm" color="gray.600">
                        Agreement ID: {agreement.a_id}
                    </Text>
                    
                    <Text fontSize="sm" color="gray.600">
                        Date: {new Date(agreement.datum).toLocaleDateString()}
                    </Text>
                    
                    <Text fontSize="lg" fontWeight="bold" color={agreement.payment_exists ? "green.600" : "blue.600"}>
                        Price: ${agreement.price}
                    </Text>
                    
                    {agreement.payment_exists && (
                        <Text fontSize="sm" color="green.600" fontWeight="bold">
                            Payment Amount: ${agreement.payment_amount}
                        </Text>
                    )}
                    
                    <Button
                        colorScheme="blue"
                        size="sm"
                        onClick={() => handleViewDocuments(agreement)}
                        marginTop="2">
                        View Documents
                    </Button>
                </Flex>
            </Box>
        );
    };

    if (loading) {
        return (
            <Flex justify="center" align="center" height="200px">
                <Text>Loading...</Text>
            </Flex>
        );
    }

    return (
        <Flex width='100%' flexDir='column' justifyContent='center'>
            <Text
                marginBottom='20px'
                textAlign='center'
                fontWeight='bold'
                fontSize='30px'
                justifySelf='center'>
                All Agreements
            </Text>
            
            <Text fontSize="md" color="gray.600" textAlign="center" mb="6">
                View all your agreements. Paid agreements are highlighted in green.
            </Text>
            
            {agreements.length === 0 ? (
                <Flex justify="center" align="center" height="200px">
                    <Text fontSize="lg" color="gray.500">
                        No agreements found.
                    </Text>
                </Flex>
            ) : (
                <SimpleGrid columns={{ base: 1, md: 2, lg: 3 }} spacing={6}>
                    {agreements.map((agreement, index) => (
                        <AgreementCard key={agreement.a_id || index} agreement={agreement} />
                    ))}
                </SimpleGrid>
            )}
        </Flex>
    );
};

export default Reports;
