import { Formik, Field, Form } from "formik";
import * as Yup from "yup";
import { Button, Text, Flex, useToast } from "@chakra-ui/react";
import { useNavigate } from "react-router-dom";

const SignupSchema = Yup.object().shape({
    email: Yup.string()
        .email("Invalid email")
        .required("This field is required!"),
    password: Yup.string()
        .required("Must enter a password!")
        .min(1, "Password is required"),
});

const ClientLogin = () => {
    const navigate = useNavigate();
    const toast = useToast();

    const loginClient = async ({ email, password }) => {
        try {
            const response = await fetch("http://localhost:3001/login/client", {
                method: "POST",
                headers: {
                    "Content-Type": "application/json",
                },
                credentials: 'include',
                body: JSON.stringify({ email, password }),
            });

            const data = await response.json();
            
            if (data.success) {
                toast({
                    title: "Login successful",
                    description: `Welcome back, ${data.user.C_Name}!`,
                    status: "success",
                    duration: 3000,
                    isClosable: true,
                });
                // Small delay to ensure session is set
                setTimeout(() => {
                    navigate("/myagreements");
                }, 100);
            } else {
                toast({
                    title: "Login failed",
                    description: data.message || "Invalid credentials",
                    status: "error",
                    duration: 3000,
                    isClosable: true,
                });
            }
        } catch (error) {
            toast({
                title: "Error",
                description: "An error occurred during login",
                status: "error",
                duration: 3000,
                isClosable: true,
            });
        }
    };

    return (
        <>
            <Text
                marginBottom='20px'
                textAlign='center'
                fontWeight='bold'
                fontSize='32px'
                justifySelf='center'>
                Client Login
            </Text>
            <Formik
                initialValues={{
                    email: "",
                    password: "",
                }}
                validationSchema={SignupSchema}
                onSubmit={async (values) => {
                    await loginClient(values);
                }}>
                {({ isSubmitting, errors, touched }) => (
                    <Form width='100%'>
                        <Flex
                            boxShadow='lg'
                            hover={{ boxShadow: "xl" }}
                            borderBottomRadius='10px'
                            bg='white'
                            borderRadius='13px'
                            transition='all 0.4s ease'
                            direction='column'
                            padding='20px'
                            width='100%'>
                            
                            <Flex padding='10px' direction='column'>
                                <Flex>
                                    <label
                                        style={{ marginRight: "15px", minWidth: "80px" }}
                                        htmlFor='email'>
                                        Email
                                    </label>
                                    <Field
                                        name='email'
                                        placeholder='client@example.com'
                                        type='email'
                                    />
                                </Flex>
                                {errors.email && touched.email ? (
                                    <Flex color='red'>{errors.email}</Flex>
                                ) : null}
                            </Flex>

                            <Flex padding='10px' direction='column'>
                                <Flex>
                                    <label
                                        style={{ marginRight: "15px", minWidth: "80px" }}
                                        htmlFor='password'>
                                        Password
                                    </label>
                                    <Field
                                        name='password'
                                        placeholder='Enter password'
                                        type='password'
                                    />
                                </Flex>
                                {errors.password && touched.password ? (
                                    <Flex color='red'>{errors.password}</Flex>
                                ) : null}
                            </Flex>

                            <Button
                                colorScheme='green'
                                type='submit'
                                disabled={isSubmitting}
                                marginTop='10px'>
                                Login as Client
                            </Button>
                        </Flex>
                    </Form>
                )}
            </Formik>
        </>
    );
};

export default ClientLogin;
