import { useState, useEffect } from "react";
import { Button, Flex, SimpleGrid, Text, Box, Badge, useToast } from "@chakra-ui/react";
import { useNavigate } from "react-router-dom";

const DealershipPayments = () => {
    const [payments, setPayments] = useState([]);
    const [loading, setLoading] = useState(true);
    const navigate = useNavigate();
    const toast = useToast();

    useEffect(() => {
        fetchPayments();
    }, []);

    const fetchPayments = async () => {
        try {
            const response = await fetch("http://localhost:3001/dealership/payments", {
                credentials: 'include'
            });

            if (response.ok) {
                const data = await response.json();
                setPayments(data);
            } else if (response.status === 401) {
                toast({
                    title: "Unauthorized",
                    description: "Please log in as a dealership to view payments",
                    status: "error",
                    duration: 3000,
                    isClosable: true,
                });
                navigate("/dealershiplogin");
            } else {
                throw new Error("Failed to fetch payments");
            }
        } catch (error) {
            toast({
                title: "Error",
                description: "Failed to load payments",
                status: "error",
                duration: 3000,
                isClosable: true,
            });
        } finally {
            setLoading(false);
        }
    };

    const handleViewPayment = (agreementId) => {
        navigate("/paymentdetails", { state: { agreementId } });
    };

    const PaymentCard = ({ payment }) => {
        return (
            <Box
                borderWidth="1px"
                borderRadius="lg"
                overflow="hidden"
                p="4"
                bg="white"
                boxShadow="md"
                _hover={{ boxShadow: "lg" }}
                transition="all 0.3s">
                
                <Flex direction="column" gap="2">
                    <Flex justify="space-between" align="center">
                        <Text fontWeight="bold" fontSize="lg">
                            {payment.make} {payment.model}
                        </Text>
                        <Badge colorScheme="green">
                            Paid
                        </Badge>
                    </Flex>
                    
                    <Text fontSize="sm" color="gray.600">
                        Year: {payment.p_year}
                    </Text>
                    
                    <Text fontSize="sm" color="gray.600">
                        Color: {payment.color}
                    </Text>
                    
                    <Text fontSize="sm" color="gray.600">
                        Client: {payment.client_name}
                    </Text>
                    
                    <Text fontSize="sm" color="gray.600">
                        Agreement ID: {payment.a_id}
                    </Text>
                    
                    <Text fontSize="sm" color="gray.600">
                        Agreement Date: {new Date(payment.agreement_date).toLocaleDateString()}
                    </Text>
                    
                    <Text fontSize="lg" fontWeight="bold" color="green.600">
                        Amount: ${payment.amount}
                    </Text>
                    
                    <Text fontSize="sm" color="gray.600">
                        Bank: {payment.bank}
                    </Text>
                    
                    <Button
                        colorScheme="blue"
                        size="sm"
                        onClick={() => handleViewPayment(payment.a_id)}
                        marginTop="2">
                        View Payment Details
                    </Button>
                </Flex>
            </Box>
        );
    };

    if (loading) {
        return (
            <Flex justify="center" align="center" height="200px">
                <Text>Loading payments...</Text>
            </Flex>
        );
    }

    return (
        <Flex width='100%' flexDir='column' justifyContent='center'>
            <Text
                marginBottom='20px'
                textAlign='center'
                fontWeight='bold'
                fontSize='30px'
                justifySelf='center'>
                Received Payments
            </Text>
            
            {payments.length === 0 ? (
                <Flex justify="center" align="center" height="200px">
                    <Text fontSize="lg" color="gray.500">
                        No payments received yet. Payments will appear here when clients make payments for your agreements.
                    </Text>
                </Flex>
            ) : (
                <SimpleGrid columns={{ base: 1, md: 2, lg: 3 }} spacing={6}>
                    {payments.map((payment, index) => (
                        <PaymentCard key={payment.p_id || index} payment={payment} />
                    ))}
                </SimpleGrid>
            )}
        </Flex>
    );
};

export default DealershipPayments;
