import { useState, useEffect } from "react";
import { Button, Flex, Box, Text, Badge, useToast } from "@chakra-ui/react";
import { useNavigate, useLocation } from "react-router-dom";

const PaymentDetails = () => {
    const [paymentDetails, setPaymentDetails] = useState(null);
    const [loading, setLoading] = useState(true);
    const navigate = useNavigate();
    const location = useLocation();
    const toast = useToast();
    
    // Get agreement ID or VIN from location state
    const { agreementId, vin } = location.state || {};

    useEffect(() => {
        if (agreementId || vin) {
            fetchPaymentDetails();
        } else {
            toast({
                title: "Error",
                description: "No agreement ID or VIN provided",
                status: "error",
                duration: 3000,
                isClosable: true,
            });
            navigate(-1);
        }
    }, [agreementId, vin]);

    const fetchPaymentDetails = async () => {
        try {
            let url;
            if (agreementId) {
                url = `http://localhost:3001/payment/${agreementId}`;
            } else if (vin) {
                url = `http://localhost:3001/payment/vin/${vin}`;
            } else {
                throw new Error("No valid identifier provided");
            }

            const response = await fetch(url, {
                credentials: 'include'
            });

            if (response.ok) {
                const data = await response.json();
                setPaymentDetails(data);
            } else {
                throw new Error("Failed to fetch payment details");
            }
        } catch (error) {
            toast({
                title: "Error",
                description: "Failed to load payment details",
                status: "error",
                duration: 3000,
                isClosable: true,
            });
            navigate(-1);
        } finally {
            setLoading(false);
        }
    };

    if (loading) {
        return (
            <Flex justify="center" align="center" height="200px">
                <Text>Loading payment details...</Text>
            </Flex>
        );
    }

    if (!paymentDetails) {
        return (
            <Flex justify="center" align="center" height="200px">
                <Text>Payment details not found</Text>
            </Flex>
        );
    }

    return (
        <Flex width='100%' flexDir='column' justifyContent='center' maxWidth='800px'>
            <Text
                marginBottom='20px'
                textAlign='center'
                fontWeight='bold'
                fontSize='30px'
                justifySelf='center'>
                Payment Details
            </Text>
            
            <Box
                borderWidth="1px"
                borderRadius="lg"
                overflow="hidden"
                p="6"
                bg="white"
                boxShadow="lg">
                
                <Flex direction="column" gap="4">
                    {/* Payment Status */}
                    <Flex justify="space-between" align="center">
                        <Text fontWeight="bold" fontSize="xl">
                            Payment Information
                        </Text>
                        <Badge colorScheme="green" fontSize="md" p="2">
                            PAID
                        </Badge>
                    </Flex>
                    
                    {/* Vehicle Information */}
                    <Box bg="gray.50" p="4" borderRadius="md">
                        <Text fontWeight="bold" mb="2">Vehicle Details:</Text>
                        <Text>Make: {paymentDetails.make}</Text>
                        <Text>Model: {paymentDetails.model}</Text>
                        <Text>Year: {paymentDetails.p_year}</Text>
                        <Text>Color: {paymentDetails.color}</Text>
                    </Box>
                    
                    {/* Agreement Information */}
                    <Box bg="blue.50" p="4" borderRadius="md">
                        <Text fontWeight="bold" mb="2">Agreement Details:</Text>
                        <Text>Agreement ID: {paymentDetails.a_id}</Text>
                        <Text>Agreement Date: {new Date(paymentDetails.agreement_date).toLocaleDateString()}</Text>
                        <Text>Agreement Price: ${paymentDetails.agreement_price}</Text>
                    </Box>
                    
                    {/* Payment Information */}
                    <Box bg="green.50" p="4" borderRadius="md">
                        <Text fontWeight="bold" mb="2">Payment Details:</Text>
                        <Text>Payment ID: {paymentDetails.p_id}</Text>
                        <Text>Amount Paid: ${paymentDetails.amount}</Text>
                        <Text>Bank: {paymentDetails.bank}</Text>
                        <Text>IBAN: {paymentDetails.iban}</Text>
                    </Box>
                    
                    {/* Parties Information */}
                    <Box bg="purple.50" p="4" borderRadius="md">
                        <Text fontWeight="bold" mb="2">Parties:</Text>
                        <Text>Dealership: {paymentDetails.dealership_name}</Text>
                        <Text>Client: {paymentDetails.client_name}</Text>
                    </Box>
                    
                    {/* Action Buttons */}
                    <Flex direction="row" gap="3" marginTop="4">
                        <Button
                            colorScheme="blue"
                            onClick={() => navigate(-1)}>
                            Back
                        </Button>
                        <Button
                            colorScheme="gray"
                            onClick={() => window.print()}>
                            Print Receipt
                        </Button>
                    </Flex>
                </Flex>
            </Box>
        </Flex>
    );
};

export default PaymentDetails;
