import { Component, OnInit } from '@angular/core';
import { MatDialog } from '@angular/material/dialog';
import { MatSnackBar, MatSnackBarRef, SimpleSnackBar } from '@angular/material/snack-bar';
import { Router } from '@angular/router';
import { IPharmacyHead, IPharmacyHeadRequest, IPharmacy } from '../shared/interfaces';
import { DataService } from '../shared/data.service';
import { EditPharmacyHeadDialogComponent } from '../dialogs/edit-pharmacy-head-dialog/edit-pharmacy-head-dialog.component';
import { PharmacyDialogComponent } from '../dialogs/pharmacy-dialog/pharmacy-dialog.component';
import { PharmacyHeadDialogComponent } from '../nav-menu/dialogs/pharmacy-head-dialog/pharmacy-head-dialog.component';


@Component({
  selector: 'app-admin',
  templateUrl: './admin.component.html',
  styleUrls: ['./admin.component.css']
})
export class AdminComponent implements OnInit {
  public heads: IPharmacyHead[] = [];
  public requests: IPharmacyHeadRequest[] = [];
  public head: IPharmacyHead;

  constructor(private dataService: DataService, private dialog: MatDialog, private snackBar: MatSnackBar, private router: Router) {

  }

  ngOnInit(): void {
    this.dataService.getPharmacyHeads()
        .subscribe((pHeads: IPharmacyHead[]) => {
          this.heads = pHeads;
        },
        (err: any) => console.log(err),
        () => console.log("PharmacyHead data retrieved"));

    this.dataService.getClaimingRequests()
    .subscribe((pRequests: IPharmacyHeadRequest[]) => {
      this.requests = pRequests;
    },
    (err: any) => console.log(err),
    () => console.log("PharmacyHead data retrieved"));
  }

  createHead() {
    this.dataService.insertPharmacyHead(this.head)
        .subscribe((cHead: IPharmacyHead) => {
          if(cHead) {
            this.heads.push(cHead);
            this.openSnackBar("New head created!","OK");
          }
        },
        (err: any) => console.log(err),
        () => console.log("PharmacyHead inserted"));
    this.head = null;
  }

  deletePharmacyHead(dHead: IPharmacyHead) {
    this.dataService.deletePharmacyHead(dHead.id)
        .subscribe((status: boolean) => {
          if(status) {
            this.openSnackBar("Head deleted!","OK");
          }
        },
        (err: any) => console.log(err),
        () => console.log("PharmacyHead deleted"));
  }

  openEditPharmacyHeadDialog(eHead: IPharmacyHead) {
    let dialogRef = this.dialog.open(EditPharmacyHeadDialogComponent, {
      width: '450px',
      data: eHead
    });
    dialogRef.afterClosed().subscribe((editedHead: IPharmacyHead) => {
      if(editedHead){
        this.heads = this.heads.filter(x => x !== eHead);
        this.heads.push(editedHead);
        this.dataService.updatePharmacyHead(editedHead)
            .subscribe((hd: IPharmacyHead) => {
              if(hd) {
                this.openSnackBar("Success! PharmacyHead edited", "OK").onAction().subscribe(() => {
                  window.location.reload();
                });
              }
              else {
                this.openSnackBar("PharmacyHead edit failed", "Try again");
              }
            },
            (err: any) => console.log(err),
            () => console.log('PharmacyHead data updated'));
      }
    });
  }

  openPharmacyDialog(pharmacy: IPharmacy): void {
    this.dialog.open(PharmacyDialogComponent, {
      width: '450px',
      data: pharmacy
    });
  }

  openPharmacyHeadDialog(hd: IPharmacyHead): void {
    this.dialog.open(PharmacyHeadDialogComponent, {
      width: '450px',
      data: hd
    });
  }

  rejectRequest(req: IPharmacyHeadRequest) {
    this.dataService.deleteClaimingRequest(req.id)
        .subscribe((status: boolean) => {
          if(status) {
            this.openSnackBar("Request rejected!","OK");
          }
          else {
            this.openSnackBar("Something went wrong","Try again");
          }
        },
        (err: any) => console.log(err),
        () => console.log("PharmacyHeadRequest deleted"));
  }

  approveRequest(req: IPharmacyHeadRequest) {
    if(req) {
    req.PharmacyHead.Pharmacy.push(req.Pharmacy);
    this.dataService.updatePharmacyHead(req.PharmacyHead)
        .subscribe(() => {
          
        },
        (err: any) => console.log(err),
        () => console.log("PharmacyHead updated"))
    }
  }

  openSnackBar(message: string, action: string) : MatSnackBarRef<SimpleSnackBar> {
    return this.snackBar.open(message, action, {
      duration: 5000,
    });
  }

}
