# Kupi.mk - Docker Setup Guide

## 🐳 Docker Deployment

This guide will help you run the entire Kupi.mk application using Docker, making it work consistently on any PC.

## Prerequisites

- Docker Desktop (or Docker Engine + Docker Compose)
- 4GB RAM minimum
- 2GB free disk space

## 🚀 Quick Start

### 1. Clone the repository
```bash
git clone <repository-url>
cd kupi-mk
```

### 2. Run the setup script
```bash
./docker-setup.sh
```

### 3. Access the application
- **Frontend**: http://localhost:3000
- **Backend API**: http://localhost:5001
- **Database**: localhost:5433

## 📁 Docker Architecture

```
kupi-mk/
├── docker-compose.yml          # Orchestrates all services
├── docker-setup.sh             # Automated setup script
├── backend/
│   ├── Dockerfile              # Backend container config
│   ├── .dockerignore           # Files to exclude
│   └── .env.docker             # Environment template
├── frontend/
│   ├── Dockerfile              # Frontend container config
│   ├── nginx.conf              # Nginx configuration
│   ├── .dockerignore           # Files to exclude
│   └── .env.docker             # Environment template
└── database_*.sql              # Database initialization
```

## 🔧 Manual Setup

If you prefer manual control:

### 1. Create environment files
```bash
cp backend/.env.docker backend/.env
cp frontend/.env.docker frontend/.env
```

### 2. Build and start services
```bash
docker-compose up --build -d
```

### 3. View logs
```bash
docker-compose logs -f
```

### 4. Stop services
```bash
docker-compose down
```

## 🌐 Services

| Service | Container Port | Host Port | Description |
|---------|----------------|-----------|-------------|
| Frontend | 80 | 3000 | React app served by Nginx |
| Backend | 5001 | 5001 | Node.js API server |
| Database | 5432 | 5433 | PostgreSQL database |

## 🔒 Environment Variables

### Backend (.env)
```env
DB_HOST=postgres
DB_PORT=5432
DB_NAME=kupi_mk
DB_USER=admin
DB_PASSWORD=password123
JWT_SECRET=your-secret-key
PORT=5001
```

### Frontend (.env)
```env
REACT_APP_API_URL=http://localhost:5001/api
REACT_APP_BACKEND_URL=http://localhost:5001
```

## 🗄️ Data Persistence

- **Database data**: Stored in Docker volume `postgres_data`
- **Uploaded files**: Stored in Docker volume `backend_uploads`
- **Data survives container restarts**

## 🛠️ Development vs Production

### Development Mode
```bash
# Start with file watching and hot reload
docker-compose -f docker-compose.dev.yml up
```

### Production Mode
```bash
# Optimized builds, no dev dependencies
docker-compose up --build -d
```

## 🔍 Troubleshooting

### Check service status
```bash
docker-compose ps
```

### View logs
```bash
# All services
docker-compose logs

# Specific service
docker-compose logs backend
docker-compose logs frontend
docker-compose logs postgres
```

### Restart services
```bash
docker-compose restart
```

### Clean rebuild
```bash
docker-compose down --volumes --remove-orphans
docker-compose up --build -d
```

### Access database directly
```bash
docker-compose exec postgres psql -U admin -d kupi_mk
```

## 🚨 Common Issues

### Port conflicts
If ports are already in use, edit `docker-compose.yml`:
```yaml
ports:
  - "3001:80"  # Change 3000 to 3001
  - "5002:5001" # Change 5001 to 5002
```

### Permission issues on Linux
```bash
sudo docker-compose up --build -d
```

### Out of disk space
```bash
docker system prune -a --volumes
```

## 📊 Monitoring

### Resource usage
```bash
docker stats
```

### Container health
```bash
docker-compose exec backend curl http://localhost:5001/api/products
```

## 🔄 Updates

### Pull latest changes
```bash
git pull
docker-compose up --build -d
```

### Database migrations
Database changes are automatically applied on container startup via initialization scripts.

## 🌍 Deployment to Cloud

This Docker setup works on:
- ✅ **AWS ECS/EKS**
- ✅ **Google Cloud Run**
- ✅ **Azure Container Instances**
- ✅ **DigitalOcean App Platform**
- ✅ **Heroku** (with Docker support)
- ✅ **Any VPS** with Docker

## 🔐 Security Notes

- Change default passwords in production
- Use environment variables for secrets
- Enable HTTPS with reverse proxy
- Regular security updates

## 📞 Support

For issues with Docker setup:
1. Check logs: `docker-compose logs`
2. Verify requirements are met
3. Try clean rebuild
4. Check firewall/antivirus settings