#!/bin/bash

# Export local database and import to Docker
echo "🔄 Copying local database to Docker container..."

# Configuration
LOCAL_DB_NAME="kupi_mk"
LOCAL_DB_USER="admin"
DOCKER_CONTAINER="kupi-mk-postgres"
EXPORT_FILE="/tmp/local-db-export.sql"

# Step 1: Export local database
echo "📤 Step 1: Exporting local database..."
pg_dump -h localhost -p 5432 -U $LOCAL_DB_USER -d $LOCAL_DB_NAME \
  --no-owner --no-acl --clean --if-exists \
  > "$EXPORT_FILE"

if [ $? -ne 0 ]; then
    echo "❌ Failed to export local database!"
    echo "Make sure:"
    echo "  1. Local PostgreSQL is running on port 5432"
    echo "  2. Database '$LOCAL_DB_NAME' exists"
    echo "  3. User '$LOCAL_DB_USER' has access"
    exit 1
fi

echo "✅ Local database exported successfully!"
echo "📁 Export size: $(du -h "$EXPORT_FILE" | cut -f1)"

# Step 2: Check if Docker container is running
echo "🐳 Step 2: Checking Docker container..."
if ! docker ps | grep -q $DOCKER_CONTAINER; then
    echo "❌ Docker container '$DOCKER_CONTAINER' is not running!"
    echo "Start it with: docker-compose up -d postgres"
    exit 1
fi

echo "✅ Docker container is running!"

# Step 3: Copy SQL file to Docker container
echo "📋 Step 3: Copying SQL file to Docker container..."
docker cp "$EXPORT_FILE" $DOCKER_CONTAINER:/tmp/import.sql

# Step 4: Import into Docker database
echo "📥 Step 4: Importing into Docker database..."
docker exec -i $DOCKER_CONTAINER psql -U admin -d kupi_mk -f /tmp/import.sql

if [ $? -eq 0 ]; then
    echo "✅ Database successfully copied to Docker!"
    echo ""
    echo "🎉 Your local database is now available in Docker container!"
    echo "   Docker database: localhost:5433"
    echo ""
    echo "🔍 To verify, you can check with:"
    echo "   docker exec -it $DOCKER_CONTAINER psql -U admin -d kupi_mk -c '\\dt'"
    
    # Cleanup
    rm "$EXPORT_FILE"
    docker exec $DOCKER_CONTAINER rm /tmp/import.sql
    
else
    echo "❌ Failed to import database to Docker!"
    echo "Check logs with: docker-compose logs postgres"
fi