import React, { useState, useEffect } from 'react';
import { useProducts } from '../context/ProductContext';
import ProductCard from '../components/ProductCard';

const Home = () => {
  const { products, loading, fetchProducts, categories, fetchCategories } = useProducts();
  const [selectedCategory, setSelectedCategory] = useState('');
  const [searchTerm, setSearchTerm] = useState('');

  useEffect(() => {
    fetchProducts();
    fetchCategories();
  }, []);

  const handleSearch = (e) => {
    e.preventDefault();
    const filters = {};
    if (selectedCategory) filters.category = selectedCategory;
    if (searchTerm) filters.search = searchTerm;
    fetchProducts(filters);
  };

  const handleCategoryChange = (categoryId) => {
    setSelectedCategory(categoryId);
    const filters = {};
    if (categoryId) filters.category = categoryId;
    if (searchTerm) filters.search = searchTerm;
    fetchProducts(filters);
  };

  return (
    <div className="min-h-screen">
      {/* Hero Section */}
      <div className="bg-gradient-to-r from-primary-600 to-primary-800 text-white py-16 mb-8 rounded-lg">
        <div className="text-center">
          <h1 className="text-4xl md:text-6xl font-bold mb-4">
            Добредојдовте на Kupi.mk
          </h1>
          <p className="text-xl md:text-2xl mb-8 opacity-90">
            Специјализирана платформа за македонски производи
          </p>
          <p className="text-lg opacity-80 max-w-2xl mx-auto">
            Откријте ги најдобрите локални производи и поддржете ги домашните претприемачи
          </p>
        </div>
      </div>

      {/* Search and Filter Section */}
      <div className="bg-white p-6 rounded-lg shadow-md mb-8">
        <form onSubmit={handleSearch} className="flex flex-col md:flex-row gap-4">
          <div className="flex-1">
            <input
              type="text"
              placeholder="Пребарај производи..."
              value={searchTerm}
              onChange={(e) => setSearchTerm(e.target.value)}
              className="input-field"
            />
          </div>
          
          <div className="md:w-64">
            <select
              value={selectedCategory}
              onChange={(e) => handleCategoryChange(e.target.value)}
              className="input-field"
            >
              <option value="">Сите категории</option>
              {categories.map(category => (
                <option key={category.id} value={category.id}>
                  {category.name}
                </option>
              ))}
            </select>
          </div>
          
          <button type="submit" className="btn-primary px-8">
            Пребарај
          </button>
        </form>
      </div>

      {/* Categories Quick Links */}
      <div className="mb-8">
        <h2 className="text-2xl font-bold text-gray-900 mb-4">Популарни категории</h2>
        <div className="grid grid-cols-2 md:grid-cols-4 lg:grid-cols-8 gap-4">
          {categories.slice(0, 8).map(category => (
            <button
              key={category.id}
              onClick={() => handleCategoryChange(category.id)}
              className={`p-4 rounded-lg text-center transition-colors ${
                selectedCategory === category.id.toString()
                  ? 'bg-primary-600 text-white'
                  : 'bg-white text-gray-700 hover:bg-gray-50 border border-gray-200'
              }`}
            >
              <div className="text-sm font-medium">
                {category.name}
              </div>
            </button>
          ))}
        </div>
      </div>

      {/* Products Grid */}
      <div className="mb-8">
        <div className="flex justify-between items-center mb-6">
          <h2 className="text-2xl font-bold text-gray-900">
            {selectedCategory ? 'Филтрирани производи' : 'Најнови производи'}
          </h2>
          <span className="text-gray-600">
            {products.length} производи
          </span>
        </div>

        {loading ? (
          <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 xl:grid-cols-4 gap-6">
            {[...Array(8)].map((_, i) => (
              <div key={i} className="bg-white rounded-lg shadow-md overflow-hidden animate-pulse">
                <div className="h-48 bg-gray-300"></div>
                <div className="p-4">
                  <div className="h-4 bg-gray-300 rounded mb-2"></div>
                  <div className="h-3 bg-gray-300 rounded mb-3"></div>
                  <div className="h-4 bg-gray-300 rounded w-1/2"></div>
                </div>
              </div>
            ))}
          </div>
        ) : products.length > 0 ? (
          <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 xl:grid-cols-4 gap-6">
            {products.map(product => (
              <ProductCard key={product.id} product={product} />
            ))}
          </div>
        ) : (
          <div className="text-center py-12">
            <div className="text-gray-500 text-lg mb-4">
              Нема најдени производи
            </div>
            <p className="text-gray-400">
              Пробајте со различни критериуми за пребарување
            </p>
          </div>
        )}
      </div>

      {/* Call to Action */}
      <div className="bg-primary-50 rounded-lg p-8 text-center">
        <h3 className="text-2xl font-bold text-gray-900 mb-4">
          Продавате производи?
        </h3>
        <p className="text-gray-600 mb-6">
          Придружете се на нашата платформа и достигнете до илјадници потенцијални купувачи
        </p>
        <button className="btn-primary text-lg px-8 py-3">
          Започнете да продавате
        </button>
      </div>
    </div>
  );
};

export default Home;
