@echo off
title Kupi.mk Database Setup for Windows
color 0A

echo ===============================================
echo    Kupi.mk Database Setup Assistant
echo ===============================================
echo.

REM Check if PostgreSQL is installed
echo [1/5] Checking PostgreSQL installation...
psql --version >nul 2>&1
if %errorlevel% equ 0 (
    echo ✓ PostgreSQL is installed
) else (
    echo ✗ PostgreSQL is not installed or not in PATH
    echo.
    echo Please install PostgreSQL first:
    echo 1. Download from: https://www.postgresql.org/download/windows/
    echo 2. Run the installer as Administrator
    echo 3. Remember the postgres user password
    echo 4. Add PostgreSQL to PATH if needed
    echo.
    pause
    exit /b 1
)

echo.
echo [2/5] Testing PostgreSQL service...
sc query postgresql-x64-15 | find "RUNNING" >nul
if %errorlevel% equ 0 (
    echo ✓ PostgreSQL service is running
) else (
    echo ! PostgreSQL service is not running
    echo Starting PostgreSQL service...
    net start postgresql-x64-15 >nul 2>&1
    if %errorlevel% equ 0 (
        echo ✓ PostgreSQL service started
    ) else (
        echo ✗ Failed to start PostgreSQL service
        echo Please start it manually from Services
        pause
        exit /b 1
    )
)

echo.
echo [3/5] Creating database and user...
echo.
echo Please enter your PostgreSQL postgres user password:
set /p postgres_password="Password: "

REM Create database setup script
echo CREATE DATABASE kupi_mk; > temp_setup.sql
echo CREATE USER kupi_user WITH PASSWORD 'kupi_password'; >> temp_setup.sql
echo GRANT ALL PRIVILEGES ON DATABASE kupi_mk TO kupi_user; >> temp_setup.sql
echo \q >> temp_setup.sql

REM Execute the setup script
echo.
echo Executing database setup...
psql -U postgres -h localhost -f temp_setup.sql

if %errorlevel% equ 0 (
    echo ✓ Database and user created successfully
) else (
    echo ✗ Failed to create database. Please check your postgres password.
    del temp_setup.sql
    pause
    exit /b 1
)

REM Clean up
del temp_setup.sql

echo.
echo [4/5] Creating .env configuration file...

REM Check if backend directory exists
if not exist "kupi-mk\backend" (
    echo ✗ Backend directory not found
    echo Please run this script from the project root directory
    pause
    exit /b 1
)

REM Create .env file
echo # Database Configuration > kupi-mk\backend\.env
echo DB_HOST=localhost >> kupi-mk\backend\.env
echo DB_PORT=5432 >> kupi-mk\backend\.env
echo DB_NAME=kupi_mk >> kupi-mk\backend\.env
echo DB_USER=kupi_user >> kupi-mk\backend\.env
echo DB_PASSWORD=kupi_password >> kupi-mk\backend\.env
echo. >> kupi-mk\backend\.env
echo # JWT Secret ^(change this to a random string^) >> kupi-mk\backend\.env
echo JWT_SECRET=your_super_secret_jwt_key_here_change_this_in_production >> kupi-mk\backend\.env
echo. >> kupi-mk\backend\.env
echo # Server Configuration >> kupi-mk\backend\.env
echo PORT=5000 >> kupi-mk\backend\.env
echo NODE_ENV=development >> kupi-mk\backend\.env
echo. >> kupi-mk\backend\.env
echo # File Upload Configuration >> kupi-mk\backend\.env
echo UPLOAD_PATH=uploads/products >> kupi-mk\backend\.env
echo MAX_FILE_SIZE=5242880 >> kupi-mk\backend\.env

echo ✓ .env file created in kupi-mk\backend\.env

echo.
echo [5/5] Testing database connection...
cd kupi-mk\backend

REM Check if node_modules exists
if not exist "node_modules" (
    echo Installing backend dependencies...
    call npm install
)

REM Create a simple test script
echo const { Pool } = require^('pg'^); > test-connection.js
echo require^('dotenv'^).config^(^); >> test-connection.js
echo. >> test-connection.js
echo const pool = new Pool^({ >> test-connection.js
echo   user: process.env.DB_USER, >> test-connection.js
echo   host: process.env.DB_HOST, >> test-connection.js
echo   database: process.env.DB_NAME, >> test-connection.js
echo   password: process.env.DB_PASSWORD, >> test-connection.js
echo   port: process.env.DB_PORT, >> test-connection.js
echo }^); >> test-connection.js
echo. >> test-connection.js
echo async function testConnection^(^) { >> test-connection.js
echo   try { >> test-connection.js
echo     const client = await pool.connect^(^); >> test-connection.js
echo     console.log^('✓ Database connected successfully!'^); >> test-connection.js
echo     client.release^(^); >> test-connection.js
echo     process.exit^(0^); >> test-connection.js
echo   } catch ^(err^) { >> test-connection.js
echo     console.error^('✗ Database connection failed:', err.message^); >> test-connection.js
echo     process.exit^(1^); >> test-connection.js
echo   } >> test-connection.js
echo } >> test-connection.js
echo. >> test-connection.js
echo testConnection^(^); >> test-connection.js

REM Test the connection
node test-connection.js

if %errorlevel% equ 0 (
    echo.
    echo ===============================================
    echo           🎉 SETUP COMPLETED SUCCESSFULLY! 🎉
    echo ===============================================
    echo.
    echo Database Configuration:
    echo - Database: kupi_mk
    echo - Username: kupi_user
    echo - Password: kupi_password
    echo - Host: localhost
    echo - Port: 5432
    echo.
    echo Next steps:
    echo 1. Start the backend: npm run dev
    echo 2. Start the frontend: cd ..\frontend ^&^& npm start
    echo 3. Open http://localhost:3000 in your browser
    echo.
    echo Configuration saved in: kupi-mk\backend\.env
    echo.
) else (
    echo.
    echo ===============================================
    echo              ⚠️ SETUP INCOMPLETE ⚠️
    echo ===============================================
    echo.
    echo Database connection test failed.
    echo Please check:
    echo 1. PostgreSQL is running
    echo 2. Postgres user password is correct
    echo 3. Firewall settings allow port 5432
    echo.
    echo For detailed troubleshooting, see:
    echo WINDOWS_DATABASE_SETUP.md
    echo.
)

REM Clean up test file
del test-connection.js

cd ..\..
echo.
echo Press any key to exit...
pause >nul
