# Kupi.mk Database Setup Script for Windows PowerShell
# Run this script as Administrator for best results

param(
    [string]$PostgresPassword = "",
    [switch]$SkipInstallCheck = $false
)

# Set console colors
$Host.UI.RawUI.BackgroundColor = "Black"
$Host.UI.RawUI.ForegroundColor = "Green"
Clear-Host

function Write-Header {
    param([string]$Message)
    Write-Host "===============================================" -ForegroundColor Cyan
    Write-Host "   $Message" -ForegroundColor Yellow
    Write-Host "===============================================" -ForegroundColor Cyan
    Write-Host ""
}

function Write-Step {
    param([string]$Step, [string]$Message)
    Write-Host "[$Step] $Message" -ForegroundColor White
}

function Write-Success {
    param([string]$Message)
    Write-Host "✓ $Message" -ForegroundColor Green
}

function Write-Error {
    param([string]$Message)
    Write-Host "✗ $Message" -ForegroundColor Red
}

function Write-Warning {
    param([string]$Message)
    Write-Host "! $Message" -ForegroundColor Yellow
}

Write-Header "Kupi.mk Database Setup Assistant"

# Step 1: Check PostgreSQL Installation
Write-Step "1/6" "Checking PostgreSQL installation..."

if (-not $SkipInstallCheck) {
    try {
        $psqlVersion = & psql --version 2>$null
        if ($LASTEXITCODE -eq 0) {
            Write-Success "PostgreSQL is installed: $psqlVersion"
        } else {
            throw "PostgreSQL not found"
        }
    } catch {
        Write-Error "PostgreSQL is not installed or not in PATH"
        Write-Host ""
        Write-Host "Please install PostgreSQL first:" -ForegroundColor Yellow
        Write-Host "1. Download from: https://www.postgresql.org/download/windows/" -ForegroundColor White
        Write-Host "2. Run the installer as Administrator" -ForegroundColor White
        Write-Host "3. Remember the postgres user password" -ForegroundColor White
        Write-Host "4. Ensure 'Add to PATH' is checked during installation" -ForegroundColor White
        Write-Host ""
        Read-Host "Press Enter to exit"
        exit 1
    }
}

# Step 2: Check PostgreSQL Service
Write-Step "2/6" "Checking PostgreSQL service..."

$pgService = Get-Service -Name "postgresql*" -ErrorAction SilentlyContinue | Select-Object -First 1

if ($pgService) {
    if ($pgService.Status -eq "Running") {
        Write-Success "PostgreSQL service is running"
    } else {
        Write-Warning "PostgreSQL service is stopped. Attempting to start..."
        try {
            Start-Service $pgService.Name
            Write-Success "PostgreSQL service started"
        } catch {
            Write-Error "Failed to start PostgreSQL service"
            Write-Host "Please start it manually from Services (services.msc)" -ForegroundColor Yellow
            Read-Host "Press Enter to exit"
            exit 1
        }
    }
} else {
    Write-Warning "PostgreSQL service not found. It might be installed differently."
}

# Step 3: Get PostgreSQL Password
Write-Step "3/6" "Getting PostgreSQL credentials..."

if ([string]::IsNullOrEmpty($PostgresPassword)) {
    Write-Host ""
    $securePassword = Read-Host "Enter your PostgreSQL 'postgres' user password" -AsSecureString
    $PostgresPassword = [Runtime.InteropServices.Marshal]::PtrToStringAuto([Runtime.InteropServices.Marshal]::SecureStringToBSTR($securePassword))
}

# Step 4: Create Database and User
Write-Step "4/6" "Creating database and user..."

$sqlCommands = @"
CREATE DATABASE kupi_mk;
CREATE USER kupi_user WITH PASSWORD 'kupi_password';
GRANT ALL PRIVILEGES ON DATABASE kupi_mk TO kupi_user;
"@

# Create temporary SQL file
$tempSqlFile = [System.IO.Path]::GetTempFileName() + ".sql"
$sqlCommands | Out-File -FilePath $tempSqlFile -Encoding UTF8

# Set PGPASSWORD environment variable
$env:PGPASSWORD = $PostgresPassword

try {
    Write-Host "Executing database setup commands..." -ForegroundColor White
    & psql -U postgres -h localhost -f $tempSqlFile 2>&1 | Out-Host
    
    if ($LASTEXITCODE -eq 0) {
        Write-Success "Database and user created successfully"
    } else {
        throw "psql command failed"
    }
} catch {
    Write-Error "Failed to create database. Please check your postgres password."
    Remove-Item $tempSqlFile -ErrorAction SilentlyContinue
    Read-Host "Press Enter to exit"
    exit 1
} finally {
    # Clean up
    Remove-Item $tempSqlFile -ErrorAction SilentlyContinue
    Remove-Item Env:PGPASSWORD -ErrorAction SilentlyContinue
}

# Step 5: Create .env Configuration
Write-Step "5/6" "Creating .env configuration file..."

$backendPath = "kupi-mk\backend"
if (-not (Test-Path $backendPath)) {
    Write-Error "Backend directory not found: $backendPath"
    Write-Host "Please run this script from the project root directory" -ForegroundColor Yellow
    Read-Host "Press Enter to exit"
    exit 1
}

$envContent = @"
# Database Configuration
DB_HOST=localhost
DB_PORT=5432
DB_NAME=kupi_mk
DB_USER=kupi_user
DB_PASSWORD=kupi_password

# JWT Secret (change this to a random string in production)
JWT_SECRET=your_super_secret_jwt_key_here_change_this_in_production

# Server Configuration
PORT=5000
NODE_ENV=development

# File Upload Configuration
UPLOAD_PATH=uploads/products
MAX_FILE_SIZE=5242880
"@

$envPath = Join-Path $backendPath ".env"
$envContent | Out-File -FilePath $envPath -Encoding UTF8

Write-Success ".env file created at: $envPath"

# Step 6: Test Database Connection
Write-Step "6/6" "Testing database connection..."

Push-Location $backendPath

# Check if node_modules exists
if (-not (Test-Path "node_modules")) {
    Write-Host "Installing backend dependencies..." -ForegroundColor White
    & npm install
    if ($LASTEXITCODE -ne 0) {
        Write-Error "Failed to install dependencies"
        Pop-Location
        Read-Host "Press Enter to exit"
        exit 1
    }
}

# Create test script
$testScript = @"
const { Pool } = require('pg');
require('dotenv').config();

const pool = new Pool({
  user: process.env.DB_USER,
  host: process.env.DB_HOST,
  database: process.env.DB_NAME,
  password: process.env.DB_PASSWORD,
  port: process.env.DB_PORT,
});

async function testConnection() {
  try {
    const client = await pool.connect();
    console.log('✓ Database connected successfully!');
    
    const result = await client.query('SELECT version()');
    console.log('PostgreSQL version:', result.rows[0].version.split(' ')[0] + ' ' + result.rows[0].version.split(' ')[1]);
    
    client.release();
    await pool.end();
    process.exit(0);
  } catch (err) {
    console.error('✗ Database connection failed:', err.message);
    process.exit(1);
  }
}

testConnection();
"@

$testScriptPath = "test-connection.js"
$testScript | Out-File -FilePath $testScriptPath -Encoding UTF8

# Run the test
Write-Host "Testing database connection..." -ForegroundColor White
& node $testScriptPath

$connectionSuccess = $LASTEXITCODE -eq 0

# Clean up
Remove-Item $testScriptPath -ErrorAction SilentlyContinue
Pop-Location

# Final Results
Write-Host ""
if ($connectionSuccess) {
    Write-Header "🎉 SETUP COMPLETED SUCCESSFULLY! 🎉"
    
    Write-Host "Database Configuration:" -ForegroundColor White
    Write-Host "- Database: kupi_mk" -ForegroundColor Gray
    Write-Host "- Username: kupi_user" -ForegroundColor Gray
    Write-Host "- Password: kupi_password" -ForegroundColor Gray
    Write-Host "- Host: localhost" -ForegroundColor Gray
    Write-Host "- Port: 5432" -ForegroundColor Gray
    Write-Host ""
    
    Write-Host "Next steps:" -ForegroundColor White
    Write-Host "1. Start the backend server:" -ForegroundColor Gray
    Write-Host "   cd kupi-mk\backend" -ForegroundColor Cyan
    Write-Host "   npm run dev" -ForegroundColor Cyan
    Write-Host ""
    Write-Host "2. Start the frontend server:" -ForegroundColor Gray
    Write-Host "   cd kupi-mk\frontend" -ForegroundColor Cyan
    Write-Host "   npm start" -ForegroundColor Cyan
    Write-Host ""
    Write-Host "3. Open your browser to:" -ForegroundColor Gray
    Write-Host "   http://localhost:3000" -ForegroundColor Cyan
    Write-Host ""
    Write-Host "Configuration saved in: kupi-mk\backend\.env" -ForegroundColor Yellow
    
} else {
    Write-Header "⚠️ SETUP INCOMPLETE ⚠️"
    
    Write-Host "Database connection test failed." -ForegroundColor Red
    Write-Host ""
    Write-Host "Please check:" -ForegroundColor White
    Write-Host "1. PostgreSQL is running" -ForegroundColor Gray
    Write-Host "2. Postgres user password is correct" -ForegroundColor Gray
    Write-Host "3. Firewall settings allow port 5432" -ForegroundColor Gray
    Write-Host "4. No other application is using port 5432" -ForegroundColor Gray
    Write-Host ""
    Write-Host "For detailed troubleshooting, see:" -ForegroundColor White
    Write-Host "WINDOWS_DATABASE_SETUP.md" -ForegroundColor Cyan
}

Write-Host ""
Read-Host "Press Enter to exit"
