# 🗄️ PostgreSQL Database Setup Guide for Windows

This guide will walk you through setting up PostgreSQL database for the Kupi.mk project on Windows.

## 📋 Prerequisites

- Windows 10 or later
- Administrator privileges on your computer
- At least 1GB of free disk space

## 🚀 Method 1: PostgreSQL Official Installer (Recommended)

### Step 1: Download PostgreSQL

1. Visit the official PostgreSQL website: https://www.postgresql.org/download/windows/
2. Click on "Download the installer"
3. Choose the latest stable version (PostgreSQL 15.x or 16.x)
4. Download the Windows x86-64 installer

### Step 2: Install PostgreSQL

1. **Run the installer** as Administrator (right-click → "Run as administrator")
2. **Installation Directory**: Keep default (`C:\Program Files\PostgreSQL\15\`)
3. **Components Selection**: Check all boxes:
   - ✅ PostgreSQL Server
   - ✅ pgAdmin 4
   - ✅ Stack Builder
   - ✅ Command Line Tools
4. **Data Directory**: Keep default (`C:\Program Files\PostgreSQL\15\data`)
5. **Password**: Enter a strong password for the `postgres` superuser
   - ⚠️ **IMPORTANT**: Remember this password! You'll need it later
   - Example: `postgres123` (use a stronger password in production)
6. **Port**: Keep default (`5432`)
7. **Advanced Options**: Keep default locale
8. **Pre Installation Summary**: Review and click "Next"
9. **Ready to Install**: Click "Next" to begin installation
10. **Installation Progress**: Wait for completion (5-10 minutes)
11. **Completing Setup**: Uncheck "Launch Stack Builder" and click "Finish"

### Step 3: Verify Installation

1. **Open Command Prompt** as Administrator
2. **Add PostgreSQL to PATH** (if not already added):
   ```cmd
   setx PATH "%PATH%;C:\Program Files\PostgreSQL\15\bin" /M
   ```
3. **Close and reopen Command Prompt**
4. **Test PostgreSQL**:
   ```cmd
   psql --version
   ```
   You should see: `psql (PostgreSQL) 15.x`

### Step 4: Create Database for Kupi.mk

1. **Open Command Prompt** as Administrator
2. **Connect to PostgreSQL**:
   ```cmd
   psql -U postgres -h localhost
   ```
3. **Enter the password** you set during installation
4. **Create the database**:
   ```sql
   CREATE DATABASE kupi_mk;
   ```
5. **Create a user for the project**:
   ```sql
   CREATE USER kupi_user WITH PASSWORD 'kupi_password';
   ```
6. **Grant privileges**:
   ```sql
   GRANT ALL PRIVILEGES ON DATABASE kupi_mk TO kupi_user;
   ```
7. **Exit PostgreSQL**:
   ```sql
   \q
   ```

### Step 5: Configure Environment Variables

1. **Create `.env` file** in `kupi-mk/backend/` folder:
   ```cmd
   cd "C:\path\to\your\project\kupi-mk\backend"
   echo. > .env
   ```

2. **Edit the `.env` file** with Notepad:
   ```cmd
   notepad .env
   ```

3. **Add the following configuration**:
   ```env
   # Database Configuration
   DB_HOST=localhost
   DB_PORT=5432
   DB_NAME=kupi_mk
   DB_USER=kupi_user
   DB_PASSWORD=kupi_password

   # JWT Secret (change this to a random string)
   JWT_SECRET=your_super_secret_jwt_key_here_change_this

   # Server Configuration
   PORT=5000
   NODE_ENV=development

   # File Upload Configuration
   UPLOAD_PATH=uploads/products
   MAX_FILE_SIZE=5242880
   ```

4. **Save and close** the file

## 🐳 Method 2: Using Docker (Alternative)

If you prefer using Docker:

### Step 1: Install Docker Desktop

1. Download Docker Desktop from: https://www.docker.com/products/docker-desktop/
2. Install and restart your computer
3. Start Docker Desktop

### Step 2: Run PostgreSQL Container

1. **Open Command Prompt**
2. **Run PostgreSQL container**:
   ```cmd
   docker run --name kupi-postgres ^
   -e POSTGRES_DB=kupi_mk ^
   -e POSTGRES_USER=kupi_user ^
   -e POSTGRES_PASSWORD=kupi_password ^
   -p 5432:5432 ^
   -d postgres:15
   ```

3. **Verify container is running**:
   ```cmd
   docker ps
   ```

### Step 3: Configure Environment

Use the same `.env` configuration as Method 1.

## 🔧 Method 3: Using pgAdmin 4 (GUI Method)

If you prefer a graphical interface:

### Step 1: Open pgAdmin 4

1. **Find pgAdmin 4** in Start Menu
2. **Launch pgAdmin 4**
3. **Set master password** (first time only)

### Step 2: Connect to PostgreSQL

1. **Right-click "Servers"** in the left panel
2. **Create > Server...**
3. **General Tab**:
   - Name: `Kupi MK Server`
4. **Connection Tab**:
   - Host: `localhost`
   - Port: `5432`
   - Username: `postgres`
   - Password: [your postgres password]
5. **Click "Save"**

### Step 3: Create Database

1. **Right-click "Databases"** under your server
2. **Create > Database...**
3. **Database name**: `kupi_mk`
4. **Click "Save"**

### Step 4: Create User

1. **Right-click "Login/Group Roles"**
2. **Create > Login/Group Role...**
3. **General Tab**:
   - Name: `kupi_user`
4. **Definition Tab**:
   - Password: `kupi_password`
5. **Privileges Tab**:
   - ✅ Can login?
   - ✅ Create databases?
6. **Click "Save"**

## 🧪 Testing the Database Connection

### Test 1: Command Line Test

```cmd
cd "C:\path\to\your\project\kupi-mk\backend"
psql -h localhost -U kupi_user -d kupi_mk
```

### Test 2: Node.js Test

1. **Create test file** `test-db.js` in backend folder:
   ```javascript
   const { Pool } = require('pg');
   require('dotenv').config();

   const pool = new Pool({
     user: process.env.DB_USER,
     host: process.env.DB_HOST,
     database: process.env.DB_NAME,
     password: process.env.DB_PASSWORD,
     port: process.env.DB_PORT,
   });

   async function testConnection() {
     try {
       const client = await pool.connect();
       console.log('✅ Database connected successfully!');
       
       const result = await client.query('SELECT NOW()');
       console.log('Current time:', result.rows[0].now);
       
       client.release();
       process.exit(0);
     } catch (err) {
       console.error('❌ Database connection failed:', err.message);
       process.exit(1);
     }
   }

   testConnection();
   ```

2. **Run the test**:
   ```cmd
   cd kupi-mk\backend
   npm install pg dotenv
   node test-db.js
   ```

## 🚨 Troubleshooting

### Problem 1: "psql is not recognized"

**Solution**: Add PostgreSQL to PATH manually
```cmd
set PATH=%PATH%;C:\Program Files\PostgreSQL\15\bin
```

### Problem 2: "Connection refused"

**Solutions**:
1. **Check if PostgreSQL service is running**:
   ```cmd
   services.msc
   ```
   Look for "postgresql-x64-15" and start it if stopped

2. **Check firewall settings**:
   - Allow PostgreSQL through Windows Firewall
   - Default port: 5432

### Problem 3: "Authentication failed"

**Solutions**:
1. **Reset postgres password**:
   ```cmd
   # Stop PostgreSQL service first
   net stop postgresql-x64-15
   
   # Start in single user mode and reset password
   # (This is complex - easier to reinstall)
   ```

2. **Or reinstall PostgreSQL** with a new password

### Problem 4: "Database does not exist"

**Solution**: Create the database again:
```sql
psql -U postgres
CREATE DATABASE kupi_mk;
```

## ✅ Final Verification Checklist

Before starting the Kupi.mk application:

- [ ] PostgreSQL is installed and running
- [ ] Database `kupi_mk` is created
- [ ] User `kupi_user` exists with proper permissions
- [ ] `.env` file is configured in `kupi-mk/backend/`
- [ ] Database connection test passes
- [ ] Node.js can connect to the database

## 🎯 Next Steps

After database setup is complete:

1. **Start the backend server**:
   ```cmd
   cd kupi-mk\backend
   npm install
   npm run dev
   ```

2. **Check backend logs** for "PostgreSQL Connected" message

3. **The database tables will be created automatically** when you start the backend for the first time

## 📞 Need Help?

If you encounter issues:

1. **Check the error messages** carefully
2. **Verify your `.env` file** configuration
3. **Ensure PostgreSQL service is running**
4. **Try restarting your computer** after installation
5. **Consider using Docker method** if installation issues persist

## 🔐 Security Notes

For production use:
- Use stronger passwords
- Restrict database access
- Use environment-specific configurations
- Enable SSL connections
- Regular backups
