import React, { useState, useEffect } from 'react';
import { useNavigate } from 'react-router-dom';
import { useCart } from '../context/CartContext';
import api from '../services/api';

const Checkout = () => {
    const { cartItems, totalPrice, clearCart } = useCart();
    const navigate = useNavigate();
    
    const [loading, setLoading] = useState(false);
    const [orderPlaced, setOrderPlaced] = useState(false);
    const [orderId, setOrderId] = useState(null);
    const [formData, setFormData] = useState({
        shippingAddress: {
            fullName: '',
            street: '',
            city: '',
            postalCode: '',
            country: 'Macedonia'
        },
        phone: '',
        paymentMethod: 'cash_on_delivery',
        notes: ''
    });

    useEffect(() => {
        // If cart is empty, redirect to home
        if (!cartItems || cartItems.length === 0) {
            navigate('/');
        }
    }, [cartItems, navigate]);

    const handleInputChange = (section, field, value) => {
        setFormData(prev => ({
            ...prev,
            [section]: {
                ...prev[section],
                [field]: value
            }
        }));
    };

    const handleSubmit = async (e) => {
        e.preventDefault();
        setLoading(true);

        try {
            const orderData = {
                shippingAddress: formData.shippingAddress,
                phone: formData.phone,
                paymentMethod: formData.paymentMethod,
                notes: formData.notes
            };

            const response = await api.post('/orders/checkout', orderData);
            
            if (response.data.success) {
                setOrderId(response.data.order.id);
                setOrderPlaced(true);
                clearCart();
                
                // Redirect to order confirmation after 3 seconds
                setTimeout(() => {
                    navigate('/orders');
                }, 3000);
            }
        } catch (error) {
            console.error('Error placing order:', error);
            alert('Failed to place order. Please try again.');
        } finally {
            setLoading(false);
        }
    };

    if (orderPlaced) {
        return (
            <div className="min-h-screen bg-gray-50 flex items-center justify-center">
                <div className="bg-white p-8 rounded-lg shadow-md text-center max-w-md">
                    <div className="mb-4">
                        <svg className="mx-auto h-16 w-16 text-green-500" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                            <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M5 13l4 4L19 7" />
                        </svg>
                    </div>
                    <h2 className="text-2xl font-bold text-gray-800 mb-2">Order Placed Successfully!</h2>
                    <p className="text-gray-600 mb-4">
                        Your order #{orderId} has been placed successfully.
                    </p>
                    <p className="text-sm text-gray-500">
                        Redirecting to order history...
                    </p>
                </div>
            </div>
        );
    }

    if (!cartItems || cartItems.length === 0) {
        return null; // This will trigger redirect in useEffect
    }

    return (
        <div className="min-h-screen bg-gray-50 py-8">
            <div className="max-w-4xl mx-auto px-4">
                <h1 className="text-3xl font-bold text-gray-800 mb-8">Checkout</h1>
                
                <div className="grid grid-cols-1 lg:grid-cols-2 gap-8">
                    {/* Order Form */}
                    <div className="bg-white p-6 rounded-lg shadow-md">
                        <h2 className="text-xl font-semibold mb-6">Order Details</h2>
                        
                        <form onSubmit={handleSubmit}>
                            {/* Shipping Address */}
                            <div className="mb-8">
                                <h3 className="text-lg font-medium mb-4">Shipping Address</h3>
                                
                                <div className="grid grid-cols-1 gap-4">
                                    <div>
                                        <label className="block text-sm font-medium text-gray-700 mb-1">
                                            Full Name *
                                        </label>
                                        <input
                                            type="text"
                                            required
                                            value={formData.shippingAddress.fullName}
                                            onChange={(e) => handleInputChange('shippingAddress', 'fullName', e.target.value)}
                                            className="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-blue-500"
                                        />
                                    </div>
                                    
                                    <div>
                                        <label className="block text-sm font-medium text-gray-700 mb-1">
                                            Street Address *
                                        </label>
                                        <input
                                            type="text"
                                            required
                                            value={formData.shippingAddress.street}
                                            onChange={(e) => handleInputChange('shippingAddress', 'street', e.target.value)}
                                            className="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-blue-500"
                                        />
                                    </div>
                                    
                                    <div className="grid grid-cols-2 gap-4">
                                        <div>
                                            <label className="block text-sm font-medium text-gray-700 mb-1">
                                                City *
                                            </label>
                                            <input
                                                type="text"
                                                required
                                                value={formData.shippingAddress.city}
                                                onChange={(e) => handleInputChange('shippingAddress', 'city', e.target.value)}
                                                className="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-blue-500"
                                            />
                                        </div>
                                        
                                        <div>
                                            <label className="block text-sm font-medium text-gray-700 mb-1">
                                                Postal Code *
                                            </label>
                                            <input
                                                type="text"
                                                required
                                                value={formData.shippingAddress.postalCode}
                                                onChange={(e) => handleInputChange('shippingAddress', 'postalCode', e.target.value)}
                                                className="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-blue-500"
                                            />
                                        </div>
                                    </div>
                                    
                                    <div>
                                        <label className="block text-sm font-medium text-gray-700 mb-1">
                                            Country
                                        </label>
                                        <select
                                            value={formData.shippingAddress.country}
                                            onChange={(e) => handleInputChange('shippingAddress', 'country', e.target.value)}
                                            className="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-blue-500"
                                        >
                                            <option value="Macedonia">Macedonia</option>
                                            <option value="Serbia">Serbia</option>
                                            <option value="Bulgaria">Bulgaria</option>
                                            <option value="Albania">Albania</option>
                                            <option value="Greece">Greece</option>
                                        </select>
                                    </div>
                                </div>
                            </div>
                            
                            {/* Contact Information */}
                            <div className="mb-8">
                                <h3 className="text-lg font-medium mb-4">Contact Information</h3>
                                
                                <div>
                                    <label className="block text-sm font-medium text-gray-700 mb-1">
                                        Phone Number *
                                    </label>
                                    <input
                                        type="tel"
                                        required
                                        value={formData.phone}
                                        onChange={(e) => setFormData(prev => ({ ...prev, phone: e.target.value }))}
                                        placeholder="e.g. +389 70 123 456"
                                        className="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-blue-500"
                                    />
                                </div>
                            </div>
                            
                            {/* Payment Method */}
                            <div className="mb-8">
                                <h3 className="text-lg font-medium mb-4">Payment Method</h3>
                                
                                <div className="space-y-3">
                                    <label className="flex items-center">
                                        <input
                                            type="radio"
                                            name="paymentMethod"
                                            value="cash_on_delivery"
                                            checked={formData.paymentMethod === 'cash_on_delivery'}
                                            onChange={(e) => setFormData(prev => ({ ...prev, paymentMethod: e.target.value }))}
                                            className="mr-3"
                                        />
                                        <span>Cash on Delivery</span>
                                    </label>
                                    
                                    <label className="flex items-center">
                                        <input
                                            type="radio"
                                            name="paymentMethod"
                                            value="bank_transfer"
                                            checked={formData.paymentMethod === 'bank_transfer'}
                                            onChange={(e) => setFormData(prev => ({ ...prev, paymentMethod: e.target.value }))}
                                            className="mr-3"
                                        />
                                        <span>Bank Transfer</span>
                                    </label>
                                </div>
                            </div>
                            
                            {/* Order Notes */}
                            <div className="mb-8">
                                <label className="block text-sm font-medium text-gray-700 mb-1">
                                    Order Notes (Optional)
                                </label>
                                <textarea
                                    rows={3}
                                    value={formData.notes}
                                    onChange={(e) => setFormData(prev => ({ ...prev, notes: e.target.value }))}
                                    placeholder="Special delivery instructions, etc."
                                    className="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-blue-500"
                                />
                            </div>
                            
                            <button
                                type="submit"
                                disabled={loading}
                                className="w-full bg-blue-600 text-white py-3 px-4 rounded-md hover:bg-blue-700 disabled:opacity-50 disabled:cursor-not-allowed"
                            >
                                {loading ? 'Placing Order...' : 'Place Order'}
                            </button>
                        </form>
                    </div>
                    
                    {/* Order Summary */}
                    <div className="bg-white p-6 rounded-lg shadow-md h-fit">
                        <h2 className="text-xl font-semibold mb-6">Order Summary</h2>
                        
                        <div className="space-y-4 mb-6">
                            {cartItems.map((item) => (
                                <div key={item.id} className="flex items-center space-x-4">
                                    <div className="w-16 h-16 bg-gray-200 rounded-md overflow-hidden">
                                        {item.images && item.images.length > 0 ? (
                                            <img 
                                                src={`http://localhost:5001${item.images[0]}`}
                                                alt={item.title}
                                                className="w-full h-full object-cover"
                                            />
                                        ) : (
                                            <div className="w-full h-full flex items-center justify-center text-gray-400">
                                                No image
                                            </div>
                                        )}
                                    </div>
                                    
                                    <div className="flex-1">
                                        <h4 className="font-medium text-gray-800">{item.title}</h4>
                                        <p className="text-sm text-gray-600">
                                            Quantity: {item.quantity}
                                        </p>
                                    </div>
                                    
                                    <div className="text-right">
                                        <p className="font-medium">${(item.price * item.quantity).toFixed(2)}</p>
                                    </div>
                                </div>
                            ))}
                        </div>
                        
                        <div className="border-t pt-4">
                            <div className="flex justify-between items-center mb-2">
                                <span className="text-gray-600">Subtotal:</span>
                                <span className="font-medium">${totalPrice.toFixed(2)}</span>
                            </div>
                            
                            <div className="flex justify-between items-center mb-2">
                                <span className="text-gray-600">Shipping:</span>
                                <span className="font-medium">Free</span>
                            </div>
                            
                            <div className="flex justify-between items-center text-lg font-bold border-t pt-2">
                                <span>Total:</span>
                                <span>${totalPrice.toFixed(2)}</span>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    );
};

export default Checkout;