// Orders (Customer) Page
// - Purpose: show the logged-in customer's order history
// - Note: seller-specific controls (status advancement) were intentionally removed from this page
//         and moved to the Seller Dashboard to maintain proper role separation.

import React, { useState, useEffect } from 'react';
import api from '../services/api';

const Orders = () => {
    const [orders, setOrders] = useState([]);
    const [loading, setLoading] = useState(true);
    const [error, setError] = useState(null);
    const [selectedOrder, setSelectedOrder] = useState(null);

    useEffect(() => {
        fetchOrders();
    }, []);

    const fetchOrders = async () => {
        try {
            const response = await api.get('/orders/history');
            if (response.data.success) {
                setOrders(response.data.orders);
            } else {
                setError('Failed to fetch orders');
            }
        } catch (error) {
            console.error('Error fetching orders:', error);
            setError('Failed to fetch orders');
        } finally {
            setLoading(false);
        }
    };

    const handleCancelOrder = async (orderId) => {
        if (!window.confirm('Are you sure you want to cancel this order?')) {
            return;
        }

        try {
            const response = await api.put(`/orders/${orderId}/status`, {
                status: 'cancelled'
            });

            if (response.data.success) {
                // Refresh orders list
                fetchOrders();
                alert('Order cancelled successfully');
            } else {
                alert(response.data.message || 'Failed to cancel order');
            }
        } catch (error) {
            console.error('Error cancelling order:', error);
            alert('Failed to cancel order. Please try again.');
        }
    };

    const getStatusColor = (status) => {
        switch (status.toLowerCase()) {
            case 'pending':
                return 'bg-yellow-100 text-yellow-800';
            case 'confirmed':
                return 'bg-blue-100 text-blue-800';
            case 'processing':
                return 'bg-purple-100 text-purple-800';
            case 'shipped':
                return 'bg-indigo-100 text-indigo-800';
            case 'delivered':
                return 'bg-green-100 text-green-800';
            case 'cancelled':
                return 'bg-red-100 text-red-800';
            default:
                return 'bg-gray-100 text-gray-800';
        }
    };

    const formatDate = (dateString) => {
        return new Date(dateString).toLocaleDateString('en-US', {
            year: 'numeric',
            month: 'long',
            day: 'numeric',
            hour: '2-digit',
            minute: '2-digit'
        });
    };

    if (loading) {
        return (
            <div className="min-h-screen bg-gray-50 flex items-center justify-center">
                <div className="text-center">
                    <div className="animate-spin rounded-full h-12 w-12 border-b-2 border-blue-600 mx-auto"></div>
                    <p className="mt-4 text-gray-600">Loading orders...</p>
                </div>
            </div>
        );
    }

    if (error) {
        return (
            <div className="min-h-screen bg-gray-50 flex items-center justify-center">
                <div className="text-center">
                    <p className="text-red-600 mb-4">{error}</p>
                    <button
                        onClick={fetchOrders}
                        className="bg-blue-600 text-white px-4 py-2 rounded-md hover:bg-blue-700"
                    >
                        Try Again
                    </button>
                </div>
            </div>
        );
    }

    return (
        <div className="min-h-screen bg-gray-50 py-8">
            <div className="max-w-6xl mx-auto px-4">
                <h1 className="text-3xl font-bold text-gray-800 mb-8">Order History</h1>
                
                {orders.length === 0 ? (
                    <div className="bg-white p-8 rounded-lg shadow-md text-center">
                        <div className="mb-4">
                            <svg className="mx-auto h-16 w-16 text-gray-400" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M16 11V7a4 4 0 00-8 0v4M5 9h14l1 12H4L5 9z" />
                            </svg>
                        </div>
                        <h2 className="text-xl font-semibold text-gray-800 mb-2">No Orders Yet</h2>
                        <p className="text-gray-600 mb-4">You haven't placed any orders yet.</p>
                        <a
                            href="/"
                            className="inline-block bg-blue-600 text-white px-6 py-2 rounded-md hover:bg-blue-700"
                        >
                            Start Shopping
                        </a>
                    </div>
                ) : (
                    <div className="space-y-6">
                        {orders.map((order) => (
                            <div key={order.id} className="bg-white rounded-lg shadow-md overflow-hidden">
                                {/* Order Header */}
                                <div className="p-6 border-b border-gray-200">
                                    <div className="flex flex-col sm:flex-row sm:items-center sm:justify-between">
                                        <div className="mb-4 sm:mb-0">
                                            <h3 className="text-lg font-semibold text-gray-800">
                                                Order #{order.id}
                                            </h3>
                                            <p className="text-sm text-gray-600">
                                                Placed on {formatDate(order.created_at)}
                                            </p>
                                        </div>
                                        
                                        <div className="flex flex-col sm:items-end">
                                            <span className={`inline-flex items-center px-3 py-1 rounded-full text-sm font-medium ${getStatusColor(order.status)}`}>
                                                {order.status.charAt(0).toUpperCase() + order.status.slice(1)}
                                            </span>
                                            <p className="mt-2 text-lg font-semibold text-gray-800">
                                                {parseFloat(order.total_amount || 0).toLocaleString('mk-MK')} ден.
                                            </p>
                                        </div>
                                    </div>
                                </div>
                                
                                {/* Order Items - Only show if selected */}
                                {selectedOrder === order.id && (
                                    <>
                                        <div className="p-6">
                                            <h4 className="font-medium text-gray-800 mb-4">Items Ordered</h4>
                                            <div className="space-y-4">
                                                {order.items && order.items.map((item, index) => (
                                                    <div key={index} className="flex items-center space-x-4">
                                                        <div className="w-16 h-16 bg-gray-200 rounded-md overflow-hidden">
                                                            {item.product_images && item.product_images.length > 0 ? (
                                                                <img 
                                                                    src={`http://localhost:5001${item.product_images[0]}`}
                                                                    alt={item.product_title}
                                                                    className="w-full h-full object-cover"
                                                                />
                                                            ) : (
                                                                <div className="w-full h-full flex items-center justify-center text-gray-400">
                                                                    No image
                                                                </div>
                                                            )}
                                                        </div>
                                                        
                                                        <div className="flex-1">
                                                            <h5 className="font-medium text-gray-800">{item.product_title}</h5>
                                                            <p className="text-sm text-gray-600">
                                                                Quantity: {item.quantity} × {parseFloat(item.price_at_time || item.price || 0).toLocaleString('mk-MK')} ден.
                                                            </p>
                                                        </div>
                                                        
                                                        <div className="text-right">
                                                            <p className="font-medium">{(parseFloat(item.price_at_time || item.price || 0) * item.quantity).toLocaleString('mk-MK')} ден.</p>
                                                        </div>
                                                    </div>
                                                ))}
                                            </div>
                                        </div>
                                        
                                        {/* Shipping Address */}
                                        {order.shipping_address && (
                                            <div className="px-6 pb-6">
                                                <h4 className="font-medium text-gray-800 mb-2">Shipping Address</h4>
                                                <div className="text-sm text-gray-600">
                                                    <p>{order.shipping_address.fullName}</p>
                                                    <p>{order.shipping_address.street}</p>
                                                    <p>{order.shipping_address.city}, {order.shipping_address.postalCode}</p>
                                                    <p>{order.shipping_address.country}</p>
                                                </div>
                                            </div>
                                        )}
                                        
                                        {/* Order Notes */}
                                        {order.notes && (
                                            <div className="px-6 pb-6">
                                                <h4 className="font-medium text-gray-800 mb-2">Order Notes</h4>
                                                <p className="text-sm text-gray-600">{order.notes}</p>
                                            </div>
                                        )}
                                    </>
                                )}
                                
                                {/* Order Actions */}
                                <div className="px-6 pb-6">
                                    <div className="flex flex-col sm:flex-row gap-3">
                                        <button
                                            onClick={() => setSelectedOrder(selectedOrder === order.id ? null : order.id)}
                                            className="flex-1 border border-gray-300 text-gray-700 py-2 px-4 rounded-md hover:bg-gray-50 transition-colors"
                                        >
                                            {selectedOrder === order.id ? 'Hide Details' : 'View Details'}
                                        </button>
                                        
                                        {order.status === 'delivered' && (
                                            <button className="flex-1 bg-blue-600 text-white py-2 px-4 rounded-md hover:bg-blue-700 transition-colors">
                                                Reorder Items
                                            </button>
                                        )}
                                        
                                        {(order.status === 'pending' || order.status === 'confirmed') && (
                                            <button 
                                                onClick={() => handleCancelOrder(order.id)}
                                                className="flex-1 bg-red-600 text-white py-2 px-4 rounded-md hover:bg-red-700 transition-colors"
                                            >
                                                Cancel Order
                                            </button>
                                        )}
                                    </div>
                                </div>
                            </div>
                        ))}
                    </div>
                )}
            </div>
        </div>
    );
};

export default Orders;