#!/usr/bin/env node

const { Pool } = require('pg');
const fs = require('fs');
const path = require('path');

// Database configuration
const pool = new Pool({
  host: process.env.DB_HOST || 'localhost',
  port: process.env.DB_PORT || 5432,
  database: process.env.DB_NAME || 'kupi_mk',
  user: process.env.DB_USER || 'admin',
  password: process.env.DB_PASSWORD || 'password123',
});

async function importDatabase(filename = 'latest-export.sql') {
  const client = await pool.connect();
  
  try {
    console.log('🔄 Starting database import...');
    
    // Check if file exists
    const exportsDir = path.join(__dirname, '../database-exports');
    const filepath = path.join(exportsDir, filename);
    
    if (!fs.existsSync(filepath)) {
      console.error(`❌ File not found: ${filepath}`);
      console.log('Available files:');
      if (fs.existsSync(exportsDir)) {
        const files = fs.readdirSync(exportsDir).filter(f => f.endsWith('.sql'));
        files.forEach(file => console.log(`  - ${file}`));
      }
      return;
    }
    
    console.log(`📁 Reading file: ${filepath}`);
    const sqlContent = fs.readFileSync(filepath, 'utf8');
    
    // Split into individual statements and execute
    const statements = sqlContent
      .split(';')
      .map(stmt => stmt.trim())
      .filter(stmt => stmt.length > 0 && !stmt.startsWith('--'));
    
    console.log(`📝 Executing ${statements.length} SQL statements...`);
    
    // Begin transaction
    await client.query('BEGIN');
    
    try {
      for (let i = 0; i < statements.length; i++) {
        const statement = statements[i];
        if (statement.trim()) {
          await client.query(statement);
          if ((i + 1) % 10 === 0) {
            console.log(`   Processed ${i + 1}/${statements.length} statements...`);
          }
        }
      }
      
      // Commit transaction
      await client.query('COMMIT');
      
      console.log('✅ Database imported successfully!');
      
      // Show statistics
      const usersCount = await client.query('SELECT COUNT(*) FROM users');
      const categoriesCount = await client.query('SELECT COUNT(*) FROM categories');
      const productsCount = await client.query('SELECT COUNT(*) FROM products');
      
      console.log('\\n📊 Import Statistics:');
      console.log(`👥 Users: ${usersCount.rows[0].count}`);
      console.log(`📂 Categories: ${categoriesCount.rows[0].count}`);
      console.log(`📦 Products: ${productsCount.rows[0].count}`);
      
    } catch (error) {
      await client.query('ROLLBACK');
      throw error;
    }
    
  } catch (error) {
    console.error('❌ Import failed:', error.message);
  } finally {
    client.release();
    await pool.end();
  }
}

// Run import if called directly
if (require.main === module) {
  const filename = process.argv[2] || 'latest-export.sql';
  importDatabase(filename);
}

module.exports = { importDatabase };
