const express = require('express');
const router = express.Router();
const { pool } = require('../config/database');
const { auth } = require('../middleware/auth');

// Get user's cart items
router.get('/', auth, async (req, res) => {
    try {
        const userId = req.user.id;
        
        const result = await pool.query(`
            SELECT 
                ci.id,
                ci.quantity,
                ci.added_at,
                p.id as product_id,
                p.title,
                p.description,
                p.price,
                p.images,
                p.stock_quantity,
                c.name as category_name
            FROM cart_items ci
            JOIN products p ON ci.product_id = p.id
            LEFT JOIN categories c ON p.category_id = c.id
            WHERE ci.user_id = $1 AND p.is_active = true
            ORDER BY ci.added_at DESC
        `, [userId]);
        
        res.json({
            success: true,
            cartItems: result.rows,
            totalItems: result.rows.reduce((sum, item) => sum + item.quantity, 0),
            totalPrice: result.rows.reduce((sum, item) => sum + (parseFloat(item.price) * item.quantity), 0)
        });
    } catch (error) {
        console.error('Error fetching cart:', error);
        res.status(500).json({ success: false, message: 'Server error' });
    }
});

// Add item to cart
router.post('/add', auth, async (req, res) => {
    try {
        const userId = req.user.id;
        const { productId, quantity = 1 } = req.body;
        
        if (!productId) {
            return res.status(400).json({ success: false, message: 'Product ID is required' });
        }
        
        if (quantity < 1) {
            return res.status(400).json({ success: false, message: 'Quantity must be at least 1' });
        }
        
        // Check if product exists and is active
        const productResult = await pool.query(
            'SELECT id, title, stock_quantity, is_active FROM products WHERE id = $1',
            [productId]
        );
        
        if (productResult.rows.length === 0) {
            return res.status(404).json({ success: false, message: 'Product not found' });
        }
        
        const product = productResult.rows[0];
        
        if (!product.is_active) {
            return res.status(400).json({ success: false, message: 'Product is not available' });
        }
        
        if (product.stock_quantity < quantity) {
            return res.status(400).json({ 
                success: false, 
                message: `Only ${product.stock_quantity} items available in stock` 
            });
        }
        
        // Check if item already exists in cart
        const existingItem = await pool.query(
            'SELECT id, quantity FROM cart_items WHERE user_id = $1 AND product_id = $2',
            [userId, productId]
        );
        
        let result;
        
        if (existingItem.rows.length > 0) {
            // Update existing item quantity
            const newQuantity = existingItem.rows[0].quantity + quantity;
            
            if (newQuantity > product.stock_quantity) {
                return res.status(400).json({ 
                    success: false, 
                    message: `Cannot add more items. Only ${product.stock_quantity} available in stock` 
                });
            }
            
            result = await pool.query(
                'UPDATE cart_items SET quantity = $1, updated_at = CURRENT_TIMESTAMP WHERE user_id = $2 AND product_id = $3 RETURNING *',
                [newQuantity, userId, productId]
            );
        } else {
            // Add new item to cart
            result = await pool.query(
                'INSERT INTO cart_items (user_id, product_id, quantity) VALUES ($1, $2, $3) RETURNING *',
                [userId, productId, quantity]
            );
        }
        
        res.json({
            success: true,
            message: `${product.title} added to cart`,
            cartItem: result.rows[0]
        });
    } catch (error) {
        console.error('Error adding to cart:', error);
        res.status(500).json({ success: false, message: 'Server error' });
    }
});

// Update cart item quantity
router.put('/update/:id', auth, async (req, res) => {
    try {
        const userId = req.user.id;
        const cartItemId = req.params.id;
        const { quantity } = req.body;
        
        if (!quantity || quantity < 1) {
            return res.status(400).json({ success: false, message: 'Quantity must be at least 1' });
        }
        
        // Check if cart item belongs to user
        const cartItemResult = await pool.query(
            'SELECT ci.*, p.stock_quantity, p.title FROM cart_items ci JOIN products p ON ci.product_id = p.id WHERE ci.id = $1 AND ci.user_id = $2',
            [cartItemId, userId]
        );
        
        if (cartItemResult.rows.length === 0) {
            return res.status(404).json({ success: false, message: 'Cart item not found' });
        }
        
        const cartItem = cartItemResult.rows[0];
        
        if (quantity > cartItem.stock_quantity) {
            return res.status(400).json({ 
                success: false, 
                message: `Only ${cartItem.stock_quantity} items available in stock` 
            });
        }
        
        const result = await pool.query(
            'UPDATE cart_items SET quantity = $1, updated_at = CURRENT_TIMESTAMP WHERE id = $2 RETURNING *',
            [quantity, cartItemId]
        );
        
        res.json({
            success: true,
            message: 'Cart updated successfully',
            cartItem: result.rows[0]
        });
    } catch (error) {
        console.error('Error updating cart:', error);
        res.status(500).json({ success: false, message: 'Server error' });
    }
});

// Remove item from cart
router.delete('/remove/:id', auth, async (req, res) => {
    try {
        const userId = req.user.id;
        const cartItemId = req.params.id;
        
        const result = await pool.query(
            'DELETE FROM cart_items WHERE id = $1 AND user_id = $2 RETURNING *',
            [cartItemId, userId]
        );
        
        if (result.rows.length === 0) {
            return res.status(404).json({ success: false, message: 'Cart item not found' });
        }
        
        res.json({
            success: true,
            message: 'Item removed from cart',
            removedItem: result.rows[0]
        });
    } catch (error) {
        console.error('Error removing from cart:', error);
        res.status(500).json({ success: false, message: 'Server error' });
    }
});

// Clear entire cart
router.delete('/clear', auth, async (req, res) => {
    try {
        const userId = req.user.id;
        
        const result = await pool.query(
            'DELETE FROM cart_items WHERE user_id = $1 RETURNING COUNT(*)',
            [userId]
        );
        
        res.json({
            success: true,
            message: 'Cart cleared successfully'
        });
    } catch (error) {
        console.error('Error clearing cart:', error);
        res.status(500).json({ success: false, message: 'Server error' });
    }
});

// Get cart summary (total items and price)
router.get('/summary', auth, async (req, res) => {
    try {
        const userId = req.user.id;
        
        const result = await pool.query(`
            SELECT 
                COUNT(ci.id) as total_items,
                COALESCE(SUM(ci.quantity), 0) as total_quantity,
                COALESCE(SUM(p.price * ci.quantity), 0) as total_price
            FROM cart_items ci
            JOIN products p ON ci.product_id = p.id
            WHERE ci.user_id = $1 AND p.is_active = true
        `, [userId]);
        
        res.json({
            success: true,
            summary: {
                totalItems: parseInt(result.rows[0].total_items),
                totalQuantity: parseInt(result.rows[0].total_quantity),
                totalPrice: parseFloat(result.rows[0].total_price)
            }
        });
    } catch (error) {
        console.error('Error fetching cart summary:', error);
        res.status(500).json({ success: false, message: 'Server error' });
    }
});

module.exports = router;