const express = require('express');
const { body, validationResult } = require('express-validator');
const { pool } = require('../config/database');
const { auth, optionalAuth } = require('../middleware/auth');
const upload = require('../middleware/upload');

const router = express.Router();

// Get all products (with optional filtering)
router.get('/', optionalAuth, async (req, res) => {
  try {
    const { category, search, page = 1, limit = 12, seller } = req.query;
    const offset = (page - 1) * limit;

    let query = `
      SELECT p.*, c.name as category_name, 
             u.username as seller_name, u.first_name, u.last_name
      FROM products p
      LEFT JOIN categories c ON p.category_id = c.id
      LEFT JOIN users u ON p.seller_id = u.id
      WHERE p.is_active = true
    `;
    const queryParams = [];
    let paramCount = 0;

    if (category) {
      paramCount++;
      query += ` AND p.category_id = $${paramCount}`;
      queryParams.push(category);
    }

    if (search) {
      paramCount++;
      query += ` AND (p.title ILIKE $${paramCount} OR p.description ILIKE $${paramCount})`;
      queryParams.push(`%${search}%`);
    }

    if (seller) {
      paramCount++;
      query += ` AND p.seller_id = $${paramCount}`;
      queryParams.push(seller);
    }

    query += ` ORDER BY p.created_at DESC LIMIT $${paramCount + 1} OFFSET $${paramCount + 2}`;
    queryParams.push(limit, offset);

    const products = await pool.query(query, queryParams);

    // Get total count for pagination
    let countQuery = `SELECT COUNT(*) FROM products p WHERE p.is_active = true`;
    const countParams = [];
    let countParamCount = 0;

    if (category) {
      countParamCount++;
      countQuery += ` AND p.category_id = $${countParamCount}`;
      countParams.push(category);
    }

    if (search) {
      countParamCount++;
      countQuery += ` AND (p.title ILIKE $${countParamCount} OR p.description ILIKE $${countParamCount})`;
      countParams.push(`%${search}%`);
    }

    if (seller) {
      countParamCount++;
      countQuery += ` AND p.seller_id = $${countParamCount}`;
      countParams.push(seller);
    }

    const totalCount = await pool.query(countQuery, countParams);

    res.json({
      products: products.rows,
      pagination: {
        currentPage: parseInt(page),
        totalPages: Math.ceil(totalCount.rows[0].count / limit),
        totalItems: parseInt(totalCount.rows[0].count),
        hasNext: page * limit < totalCount.rows[0].count,
        hasPrev: page > 1
      }
    });

  } catch (err) {
    console.error(err);
    res.status(500).json({ message: 'Server error' });
  }
});

// Get single product
router.get('/:id', optionalAuth, async (req, res) => {
  try {
    const { id } = req.params;

    const product = await pool.query(`
      SELECT p.*, c.name as category_name, 
             u.username as seller_name, u.first_name, u.last_name, u.phone, u.email
      FROM products p
      LEFT JOIN categories c ON p.category_id = c.id
      LEFT JOIN users u ON p.seller_id = u.id
      WHERE p.id = $1 AND p.is_active = true
    `, [id]);

    if (product.rows.length === 0) {
      return res.status(404).json({ message: 'Product not found' });
    }

    res.json(product.rows[0]);

  } catch (err) {
    console.error(err);
    res.status(500).json({ message: 'Server error' });
  }
});

// Create product (requires authentication)
router.post('/', auth, upload.array('images', 5), [
  body('title').notEmpty().withMessage('Title is required'),
  body('description').notEmpty().withMessage('Description is required'),
  body('price').isDecimal({ gt: 0 }).withMessage('Price must be a positive number'),
  body('categoryId').isInt().withMessage('Category is required'),
  body('stockQuantity').isInt({ min: 0 }).withMessage('Stock quantity must be a non-negative number')
], async (req, res) => {
  try {
    const errors = validationResult(req);
    if (!errors.isEmpty()) {
      return res.status(400).json({ errors: errors.array() });
    }

    const { title, description, price, categoryId, stockQuantity, location } = req.body;
    const sellerId = req.user.userId;

    // Handle uploaded images
    const imageUrls = req.files ? req.files.map(file => `/uploads/products/${file.filename}`) : [];

    const newProduct = await pool.query(`
      INSERT INTO products (title, description, price, category_id, seller_id, images, stock_quantity, location)
      VALUES ($1, $2, $3, $4, $5, $6, $7, $8)
      RETURNING *
    `, [title, description, price, categoryId, sellerId, imageUrls, stockQuantity, location]);

    res.status(201).json({
      message: 'Product created successfully',
      product: newProduct.rows[0]
    });

  } catch (err) {
    console.error(err);
    res.status(500).json({ message: 'Server error' });
  }
});

// Update product (requires authentication and ownership)
router.put('/:id', auth, upload.array('images', 5), async (req, res) => {
  try {
    const { id } = req.params;
    const { title, description, price, categoryId, stockQuantity, location, existingImages } = req.body;
    const sellerId = req.user.userId;

    // Check if product exists and belongs to user
    const existingProduct = await pool.query(
      'SELECT * FROM products WHERE id = $1 AND seller_id = $2',
      [id, sellerId]
    );

    if (existingProduct.rows.length === 0) {
      return res.status(404).json({ message: 'Product not found or unauthorized' });
    }

    // Handle images
    let finalImages = [];
    
    // Add existing images that weren't removed
    if (existingImages) {
      try {
        const parsedExistingImages = JSON.parse(existingImages);
        if (Array.isArray(parsedExistingImages)) {
          finalImages = [...parsedExistingImages];
        }
      } catch (e) {
        // If parsing fails, use existing images from database
        finalImages = existingProduct.rows[0].images || [];
      }
    }
    
    // Add new uploaded images
    if (req.files && req.files.length > 0) {
      const newImages = req.files.map(file => `/uploads/products/${file.filename}`);
      finalImages = [...finalImages, ...newImages];
    }

    // Ensure we don't exceed 5 images
    if (finalImages.length > 5) {
      finalImages = finalImages.slice(0, 5);
    }

    const updatedProduct = await pool.query(`
      UPDATE products 
      SET title = $1, description = $2, price = $3, category_id = $4, 
          images = $5, stock_quantity = $6, location = $7, updated_at = CURRENT_TIMESTAMP
      WHERE id = $8 AND seller_id = $9
      RETURNING *
    `, [title, description, price, categoryId, finalImages, stockQuantity, location, id, sellerId]);

    res.json({
      message: 'Product updated successfully',
      product: updatedProduct.rows[0]
    });

  } catch (err) {
    console.error(err);
    res.status(500).json({ message: 'Server error' });
  }
});

// Delete product (requires authentication and ownership)
router.delete('/:id', auth, async (req, res) => {
  try {
    const { id } = req.params;
    const sellerId = req.user.userId;

    const deletedProduct = await pool.query(
      'UPDATE products SET is_active = false WHERE id = $1 AND seller_id = $2 RETURNING *',
      [id, sellerId]
    );

    if (deletedProduct.rows.length === 0) {
      return res.status(404).json({ message: 'Product not found or unauthorized' });
    }

    res.json({ message: 'Product deleted successfully' });

  } catch (err) {
    console.error(err);
    res.status(500).json({ message: 'Server error' });
  }
});

// Get categories
router.get('/categories/all', async (req, res) => {
  try {
    const categories = await pool.query('SELECT * FROM categories ORDER BY name');
    res.json(categories.rows);
  } catch (err) {
    console.error(err);
    res.status(500).json({ message: 'Server error' });
  }
});

module.exports = router;
