<?php
namespace App\Models;

use Illuminate\Database\Eloquent\Concerns\HasUuids;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\BelongsToMany;
use Illuminate\Database\Eloquent\Relations\HasOne;

class Order extends Model
{
    use HasUuids;

    protected $primaryKey = 'id';
    public $incrementing = false;
    protected $keyType = 'string';

    protected $fillable = [
        'date',
        'status',
        'estimated_delivery_date',
        'buyer_id',
        'receiver_id',
        'transport_id',
        // Remove payment_id from here
    ];

    protected $attributes = [
        'status' => 'pending',
    ];

    protected $casts = [
        'date' => 'date',
        'estimated_delivery_date' => 'date',
    ];

    public static function getValidationRules($id = null): array
    {
        return [
            'date' => 'required|date|before_or_equal:today',
            'status' => 'required|string|in:pending,confirmed,processing,shipped,delivered,cancelled',
            'estimated_delivery_date' => 'required|date|after_or_equal:date',
            'buyer_id' => 'required|string|exists:clients,id',
            'receiver_id' => 'required|string|exists:clients,id',
            'transport_id' => 'nullable|string|exists:transports,id',
        ];
    }

    public function buyer(): BelongsTo
    {
        return $this->belongsTo(Client::class, 'buyer_id');
    }

    public function receiver(): BelongsTo
    {
        return $this->belongsTo(Client::class, 'receiver_id');
    }

    // Simple one-to-one relationship
    public function payment(): HasOne
    {
        return $this->hasOne(Payment::class, 'order_id');
    }

    public function transport(): BelongsTo
    {
        return $this->belongsTo(Transport::class, 'transport_id');
    }

    public function batches(): BelongsToMany
    {
        return $this->belongsToMany(Batch::class, 'order_batches')
            ->withPivot('quantity', 'price_per_unit', 'total_price')
            ->withTimestamps();
    }

    public function getTotalAmountAttribute(): float
    {
        return $this->batches->sum('pivot.total_price') ?? 0;
    }
}
